<?php

namespace Inside\Permission\Exodus\Actions\PermissionLogic\User;

use Inside\Authentication\Models\User;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Content;
use Inside\Content\Models\Contents\Users;
use Inside\Host\Exodus\Services\ContentTypeStatusService;
use Inside\Permission\Exodus\Dto\Privileges\ContentPrivilegeDto;
use Inside\Permission\Exodus\Enums\BackofficeEnum;
use Inside\Permission\Exodus\Enums\CapabilityEnum;
use Inside\Permission\Exodus\Models\Capability;
use Inside\Permission\Exodus\Services\RolePrivilegesService;

class CanCreateContent
{
    public function handle(User $user, Content $content): bool
    {
        // Check if the user is super admin
        if ($user->isSuperAdmin()) {
            return true;
        }

        // Users content is not permissible, the user must have backoffice access to create a user.
        if ($content instanceof Users) {
            return $user->hasBackofficeAccessTo('user');
        }

        // Check if the content type is permissible, if not, it's not possible to create the content.
        if (! $content::isPermissible()) {
            return false;
        }

        // Check if the user has permission to create the content type.
        if (! $user->hasContentTypePrivilegeTo(CapabilityEnum::CREATE, $content)) {
            return false;
        }

        // Retrieve every categorizable content allowed to be assigned for the current user.
        // We keep only the types of content, because we know than he got a least one content of this type allowed for assignation.
        $categorizableContentsTypeAllowed = $user
            ->getAuthorizedCategorizableContentPrivileges()
            ->map(fn (ContentPrivilegeDto $dto) => $dto->toArray())
            ->where('capability.name', CapabilityEnum::ASSIGN)
            ->pluck('index.type')
            ->unique();

        // The User must have permission to assign at least one content for each required categorizable relation to create a new content of this type.
        // We already know the list of content type allowed for assignation, so we can check if the wanted content type do not have any relation that is not allowed.
        return $content::getRequiredCategorizableRelations()
            ->values()
            ->map(fn ($relation) => type_to_class($relation))
            ->diff($categorizableContentsTypeAllowed)
            ->isEmpty();
    }
}
