<?php

namespace Inside\Permission\Console;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Artisan;
use Inside\Permission\Facades\Permission;
use Inside\Permission\Services\PermissionService;

/**
 * Rebuild dynamic model command.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class PermissionsV2InitializeCommand extends Command
{
    /**
     * @var string
     */
    protected $name = 'permissions:V2:initialize';

    /**
     * @var string
     */
    protected $description = 'Migrate old permissions to v2, you got the choice to switch between v1 and v2';

    protected array $files = [
        'vendor/maecia/inside/permission/database/migrations/2024_06_21_000001_convert_old_permissions_to_privileges.php',
        'vendor/maecia/inside/permission/database/migrations/2024_10_15_000000_remove_duplicate_authenticated_privileges.php',
        'vendor/maecia/inside/permission/database/migrations/2024_10_16_000000_compute_restriction_all_roles.php',
    ];

    /**
     * @throws \Exception
     */
    public function handle(): void
    {
        if (PermissionService::isSystemV2Initialized()) {
            $this->error('You have already initialized the system v2.');

            return;
        }

        $this->info('Inside is in maintenance mode, please wait...');
        Artisan::call('inside:down');

        setting('permission', ['system_v2_initialized' => true]);

        $this->info('Initializing system v2, it will take a while...');

        foreach ($this->files as $file) {
            if (file_exists($file)) {
                $this->info("Migrating file: $file");

                $command = collect([
                    'php',
                    '-d memory_limit=-1',
                    '-d max_execution_time=-1',
                    'artisan',
                    'migrate:refresh',
                    '--path='.$file,
                    '--force',
                ])->join(' ');

                exec(windows_os() ? str_replace('/', '\\', $command) : $command);
            }
        }

        $this->info('Migration to v2 completed successfully.');

        $this->info('Start inside:update command to update the system.');
        Artisan::call('inside:update');
    }
}
