<?php

namespace Inside\Notify\Services;

use Exception;
use Inside\Notify\Exceptions\UnknownSmsDriverException;
use Inside\Notify\Messages\SmsMessage;
use Inside\Notify\Notifications\InsideNotification;
use NumberFormatter;

class SmsService
{
    /**
     * @var string
     */
    protected $defaultDriver;

    public function __construct(string $defaultDriver)
    {
        $this->defaultDriver = $defaultDriver;
    }

    /**
     * @return string
     */
    public function getDefaultDriver(): string
    {
        return $this->defaultDriver;
    }

    /**
     * @param  string|null  $driver
     * @return SmsDriverServiceInterface
     */
    public function getDriver(?string $driver = null): SmsDriverServiceInterface
    {
        if (is_null($driver)) {
            $driver = $this->getDefaultDriver();
        }
        switch ($driver) {
            case 'nexmo':
                // TODO
                break;
            case 'sendinblue':
                return app(SendinBlueService::class);
            case 'twilio':
                return app(TwilioService::class);
            default:
                throw UnknownSmsDriverException::named($driver);
        }
        throw UnknownSmsDriverException::named($driver);
    }

    /**
     * is driver $driver available ?
     *
     * @param  string|null  $driver
     * @return bool
     */
    public function isAvailable(?string $driver = null): bool
    {
        return $this->getDriver($driver)->isAvailable();
    }

    /**
     * @param  string|null  $driver
     * @return bool
     */
    public function overQuota(?string $driver = null): bool
    {
        return $this->getDriver($driver)->overQuota();
    }

    /**
     * Get current quota
     *
     * @param  string|null  $driver
     * @return int|null
     */
    public function getQuota(?string $driver = null): ?int
    {
        return $this->getDriver($driver)->getQuota();
    }

    /**
     * Get sms segments used for the current month
     *
     * @param  string|null  $driver
     * @return int
     */
    public function getSmsCountForCurrentMonth(?string $driver = null): int
    {
        return $this->getDriver($driver)->getSmsCountForCurrentMonth();
    }

    /**
     * Get remaining credits for this month
     *
     * @param  string|null  $driver
     * @return float
     */
    public function getRemainingCreditsForCurrentPlan(?string $driver = null): float
    {
        return (float) $this->getDriver($driver)->getRemainingCreditsForCurrentPlan();
    }

    /**
     * Get getFormattedRemainingCredits
     *
     * @param  string|null  $driver
     * @param  string|null  $locale
     * @return string
     */
    public function getFormattedRemainingCredits(?string $driver = null, ?string $locale = null): string
    {
        if (is_null($locale)) {
            $locale = app()->getLocale();
        }
        try {
            $formatter = new NumberFormatter($locale, NumberFormatter::CURRENCY);

            return (string) $formatter->formatCurrency(
                (float) $this->getDriver($driver)->getRemainingCreditsForCurrentPlan(),
                (string) $this->getDriver($driver)->getCurrency()
            );
        } catch (Exception $exception) {
            return '';
        }
    }

    /**
     * Send $message
     *
     * @param SmsMessage $message
     * @param string|null $driver
     * @return bool
     */
    public function send(SmsMessage $message, ?string $driver = null): bool
    {
        return $this->getDriver($driver)->send($message);
    }

    /**
     * Prepare a sms message from a notification
     *
     * @param  InsideNotification  $notification
     * @param  string|null  $driver
     * @return mixed
     */
    public function getMessageFromNotification(InsideNotification $notification, ?string $driver = null)
    {
        return $this->getDriver($driver)->getMessageFromNotification($notification);
    }
}
