<?php

declare(strict_types=1);

namespace Inside\Notify\Services;

use Illuminate\Support\Facades\Log;
use Inside\Facades\Http;
use Inside\Support\Http\Client\Response;

final class OneSignalClientService
{
    public const API_URL = 'https://api.onesignal.com';

    public const ENDPOINT_NOTIFICATIONS = '/notifications';

    public function __construct(
        private ?string $appId = null,
        private ?string $restApiKey = null,
        private bool $isMobileActivated = false,
        private bool $isDesktopActivated = false)
    {
    }

    public function sendNotificationToPlayers(
        array $playerIds,
        string $body,
        ?string $title = null,
        ?string $url = null,
        ?string $schedule = null,
        ?array $additionalData = []
    ): mixed {
        $contents = [
            'en' => $body,
        ];

        $params = [
            'app_id' => $this->appId,
            'contents' => $contents,
            'include_external_user_ids' => $playerIds,
            'data' => $additionalData,
        ];

        if ($additionalData) {
            $params['include_external_user_ids'] = $playerIds;
        } else {
            $params['include_subscription_ids'] = $playerIds;
        }

        if (isset($url)) {
            $params['url'] = $url;
        }

        if (isset($schedule)) {
            $params['send_after'] = $schedule;
        }

        if (isset($title)) {
            $params['headings'] = [
                'en' => $title,
            ];
        }

        /** @var Response $result */
        $result = Http::withToken($this->restApiKey)->post(self::API_URL.self::ENDPOINT_NOTIFICATIONS, $params);
        if ($result->failed()) {
            Log::error(__(
                '[OneSignalClientService::sendNotificationToPlayers] failed :reason',
                [
                    'reason' => $result->body(),
                ]
            ));
        }

        return $result->json();
    }

    public function getAppId(): ?string
    {
        return $this->appId;
    }

    public function isAvailable(): bool
    {
        return ! is_null($this->appId) && ! is_null($this->restApiKey);
    }

    public function isMobileActivated(): bool
    {
        return $this->isMobileActivated;
    }

    public function isDesktopActivated(): bool
    {
        return $this->isDesktopActivated;
    }
}
