<?php

namespace Inside\Notify\Services;

use Exception;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Lang;
use Inside\Content\Models\Contents\Users;
use Inside\Notify\Exceptions\NotificationTypeValidatorException;
use Inside\Notify\Models\NotificationSubscriber;
use Inside\Notify\Models\NotificationType;
use Inside\Notify\Validators\NotificationTypeValidator;
use Inside\Permission\Facades\Permission;
use Inside\Permission\Facades\Role;

/**
 * Notification Type service.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class NotificationTypeService
{
    /**
     * Validate notification type format
     *
     * @param  array  $data
     * @throws NotificationTypeValidatorException
     */
    public function validate(array $data): void
    {
        $validator = new NotificationTypeValidator();
        $validator->validate($data);
    }

    /**
     * List notification types
     *
     * @param  Request  $request
     * @return LengthAwarePaginator|Builder[]|Collection|\Illuminate\Support\Collection|NotificationType[]
     */
    public function list(Request $request)
    {
        $filters = [];
        $query = NotificationType::query();

        if ($request->has('filters')) {
            $filters = json_decode($request->get('filters'), true) ?? [];
        }
        $langcode = list_languages()[0];
        $user = Auth::user();
        if ($user) {
            $user = Users::find($user->uuid);
            if ($user) {
                $langcode = $user->langcode;
            }
        }
        $limit = null;
        $paginate = array_key_exists('paginate', $filters) && $filters['paginate'];

        foreach ($filters as $filter => $value) {
            switch ($filter) {
                case 'paginate':
                    break;
                case 'limit':
                    if (! $paginate) {
                        $query->limit($value);
                    } else {
                        $limit = $value;
                    }
                    break;
                case 'offset':
                    if (! $paginate) {
                        $query->offset($value);
                    }
                    break;
                case 'profile':
                    $query->where($filter, $value);
                    $user = Permission::user();
                    if ($user) {
                        $query->where(function ($query) use ($user) {
                            $query->where('role', false)
                                ->orWhereHas(
                                    'roles',
                                    function ($roles) use ($user) {
                                        $roles->whereIn('inside_roles.id', Role::listUserRoleIds($user));
                                    }
                                );
                        });
                    }
                    break;
                default:
                    $query->where($filter, $value);
                    break;
            }
        }
        if ($request->get('with_subscribers', false) && is_maecia_admin()) {
            $query->with(['subscribers', 'subscribers.information']);
        }
        if ($request->get('with_subscribers_count', false) && is_maecia_admin()) {
            $query->withCount('subscribers');
        }
        // Check user is subscribed
        $query->addSelect([
            'subscribed' => NotificationSubscriber::selectRaw('case when count(*) > 0 then 1 else 0 end')
                ->where('inside_notifications_subscribers.user_uuid', $user->uuid)
                ->whereColumn('inside_notifications_types.id', '=', 'inside_notifications_subscribers.notification_type_id'),
        ]);

        if ($paginate) {
            $types = $query->paginate($limit);
        } else {
            $types = $query->get();
        }
        foreach ($types as &$type) { // @phpstan-ignore-line
            $data = $type->data;
            if (isset($data['title'])) {
                $translated = Lang::get($data['title'], [], $langcode);
                if ($translated == $data['title']) {
                    $translated = Lang::get($data['title'], [], $langcode);
                    if ($translated == $data['title']) {
                        $translated = null;
                    }
                }
                if ($translated !== null) {
                    $data['title'] = $translated;
                }
            }
            $type->data = $data;
        }

        return $types;
    }

    /**
     * Get a notification type
     *
     * @param int $notificationTypeId
     * @return NotificationType|null
     */
    public function get(int $notificationTypeId): ?NotificationType
    {
        $notificationType = NotificationType::find($notificationTypeId);
        if ($notificationType) {
            $user = Auth::user();
            $notificationType->subscribed = NotificationSubscriber::where(
                [
                    'user_uuid' => $user->uuid,
                    'notification_type_id' => $notificationType->id,
                ]
            )->count();
        }

        return $notificationType;
    }

    /**
     * Create a new notification type
     *
     * @param  array  $data
     * @return NotificationType
     * @throws NotificationTypeValidatorException
     */
    public function create(array $data): ?NotificationType
    {
        $this->validate($data);

        $notificationType = NotificationType::where(array_except($data, ['data']))->first();

        if (! $notificationType) {
            $notificationType = NotificationType::create($data);
        }

        return $notificationType;
    }

    /**
     * Update a new notification type
     *
     * @param  array  $data
     * @param  int  $notificationTypeId
     * @return NotificationType
     * @throws NotificationTypeValidatorException
     */
    public function update(array $data, int $notificationTypeId): ?NotificationType
    {
        $this->validate($data);

        $notificationType = $this->get($notificationTypeId);

        if ($notificationType) {
            $notificationType->update($data);
        }

        return $notificationType;
    }

    /**
     * Delete a notification type
     *
     * @param  int  $notificationTypeId
     * @throws Exception
     */
    public function delete(int $notificationTypeId): void
    {
        $notificationType = NotificationType::find($notificationTypeId);

        if ($notificationType) {
            $notificationType->delete();
        }
    }
}
