<?php

namespace Inside\Notify\Services;

use Exception;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\Users;
use Inside\Notify\Models\NotificationSubscriber;
use Inside\Notify\Models\NotificationType;

/**
 * Notification Subscriber service.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class NotificationSubscriberService
{
    /**
     * Subscribes $uuids to notificationType
     *
     * @param  NotificationType  $notificationType
     * @param array                                  $uuids
     */
    public function addSubscribers(NotificationType $notificationType, array $uuids): void
    {
        $users = User::find($uuids);

        $subscribers = $notificationType->subscribers;

        $notificationType->subscribers()->sync($subscribers->merge($users));
    }

    /**
     * Unsubscribes $uuids to notificationType
     *
     * @param  NotificationType  $notificationType
     * @param array                                  $uuids
     */
    public function removeSubscribers(NotificationType $notificationType, array $uuids): void
    {
        $users = User::find($uuids);

        $subscribers = $notificationType->subscribers;

        $notificationType->subscribers()->sync($subscribers->diff($users));
    }

    /**
     * @param  Request  $request
     * @param  NotificationType  $notificationType
     * @return Collection|LengthAwarePaginator
     */
    public function listSubscribers(Request $request, NotificationType $notificationType)
    {
        $filters = [];
        $query = $notificationType->subscribers();

        if ($request->has('filters')) {
            $filters = json_decode($request->get('filters'), true) ?? [];
        }
        $limit = null;
        $paginate = array_key_exists('paginate', $filters) && $filters['paginate'];

        foreach ($filters as $filter => $value) {
            switch ($filter) {
                case 'paginate':
                    break;
                case 'limit':
                    if (! $paginate) {
                        $query->limit($value);
                    } else {
                        $limit = $value;
                    }
                    break;
                case 'offset':
                    if (! $paginate) {
                        $query->offset($value);
                    }
                    break;
                default:
                    $query->where($filter, $value);
                    break;
            }
        }
        $query->with(['information']);
        if ($paginate) {
            return $query->paginate($limit);
        }

        return $query->get();
    }

    /**
     * Subscribe user to notification types
     *
     * @param  int  $notificationTypeId
     * @return bool
     */
    public function subscribe(int $notificationTypeId): bool
    {
        $user = Auth::user();

        if (NotificationSubscriber::where(
            [
                'user_uuid'            => $user->uuid,
                'notification_type_id' => $notificationTypeId,
            ]
        )->doesntExist()
        ) {
            return null !== NotificationSubscriber::create(
                [
                    'user_uuid'            => $user->uuid,
                    'notification_type_id' => $notificationTypeId,
                ]
            );
        }

        return false;
    }

    /**
     * Unsubscribe user from notification types
     *
     * @param  int  $notificationTypeId
     * @throws Exception
     */
    public function unsubscribe(int $notificationTypeId): bool
    {
        $user = Auth::user();

        return NotificationSubscriber::where(
            [
                'user_uuid'            => $user->uuid,
                'notification_type_id' => $notificationTypeId,
            ]
        )->delete();
    }
}
