<?php

namespace Inside\Notify\Notifications;

use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Lang;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Content;
use Inside\Content\Models\Contents\Users;
use Inside\Content\Transformers\ContentTransformer;
use Inside\Notify\Channels\OneSignalChannel;
use Inside\Notify\Models\NotificationPushDevice;
use Inside\Notify\Models\NotificationType;

class PushNotification extends InsideNotification
{
    protected string $deviceType;

    private ContentTransformer $transformer;

    public function __construct(
        protected NotificationType $notificationType,
        protected ?User $user,
        protected ?Content $model,
        protected ?array $data = [],
    ) {
        parent::__construct($notificationType, $user, $model, $data);

        $this->deviceType = substr(
            (string) $notificationType->via,
            strpos((string) $notificationType->via, '_') + 1
        ); //push_mobile => mobile

        $this->transformer = new ContentTransformer();
    }

    public function via(): array
    {
        if (is_null($this->user)) {
            return [];
        }

        return [OneSignalChannel::class];
    }

    /**
     * @param  mixed  $notifiable
     * @return array
     * @throws Exception
     */
    public function toOneSignal($notifiable): array
    {
        if (is_null($this->user)) {
            throw new Exception('PushNotification needs a user recipient');
        }

        $translationData = $this->getTranslationData();

        $title = $this->getTranslation(
            $this->notificationType->data['push']['subject'], // @phpstan-ignore-line
            $translationData,
            $this->user->langcode ?? config('app.locale')
        );

        $body = $this->getTranslation(
            $this->notificationType->data['push']['content'], // @phpstan-ignore-line
            $translationData,
            $this->user->langcode ?? config('app.locale')
        );

        $payload = [
            'body' => $body,
            'title' => $title,
            'url' => $this->getContentUrl(),
            'playerIds' => $this->deviceType === 'desktop'
                ? NotificationPushDevice::query()
                    ->where('user_uuid', '=', $this->user->uuid)
                    ->where('type', '=', $this->deviceType)
                    ->pluck('onesignal_player_id')
                    ->toArray()
                : [$this->user->uuid],
        ];

        if ($this->deviceType !== 'desktop') {
            $payload['additional_data'] = [
                'deep-link' => 'inside://'.$this->getContentUrl(),
            ];
        }

        return $payload;
    }

    /**
     * @return string
     */
    public function getUrl(): string
    {
        $url = $this->getContentUrl();

        return $url ? url($url) : '#';
    }

    /**
     * @return array
     * @throws Exception
     */
    public function getTranslationData(): array
    {
        if (is_null($this->user)) {
            throw new Exception('PushNotification needs a user recipient');
        }
        $data = $this->data ?? [];

        $recipient = $this->user->information;
        if (! $recipient instanceof Content) {
            throw new Exception('PushNotification needs a user recipient');
        }

        $data['to'] = [
            'uuid' => $recipient->uuid,
            'lastname' => $recipient->lastname,
            'firstname' => $recipient->firstname,
            'image' => $recipient->image,
        ];

        if (array_key_exists('from', $data)) {
            // If data from is set, let's get user infos
            try {
                $from = Users::find($data['from']);
                $data['author'] = [
                    'uuid' => $from->uuid,
                    'lastname' => $from->lastname,
                    'firstname' => $from->firstname,
                    'image' => $from->image,
                    'fullname' => "{$from->firstname} {$from->lastname}",
                ];
            } catch (ModelNotFoundException $exception) {
            }
        }

        if (isset($this->notificationType->data['fields'])) {
            $data['model'] = $this->transformer->transform($this->model, $this->notificationType->data['fields']);
        }

        $data = Arr::dot($data);

        return array_map('trim', $data); // Remove \n at end
    }

    /**
     * @param string $toTranslate
     * @param array $translationData
     * @param string $locale
     * @return string
     */
    public function getTranslation(string $toTranslate, array $translationData, string $locale): string
    {
        if (is_null($this->model)) {
            return '';
        }
        $translationKey = $this->getTranslationKey($toTranslate, $this->model, []);
        $translated = strip_tags(Lang::get($translationKey, $translationData, $locale));

        return $translated;
    }
}
