<?php

namespace Inside\Notify\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Notifications\Notification;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Carbon;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Content;
use Inside\Notify\Models\NotificationType;
use Inside\Notify\Services\HasSpecialLanguageKeys;

class InsideNotification extends Notification implements ShouldQueue
{
    use Queueable;
    use SerializesModels;
    use HasSpecialUrl;
    use HasSpecialLanguageKeys;

    private const DEFAULT_DATE_FORMAT = 'd/m/Y';

    private const DATE_FIELDS = [
        'created_at',
        'updated_at',
        'published_at',
    ];

    public function __construct(
        protected NotificationType $notificationType,
        protected ?User $user,
        protected ?Content $model,
        protected ?array $data = []
    ) {
    }

    /**
     * On tente de remplacer les dernières variables non connu par le transformer.
     * Si on y arrive pas on remplace par une chaine vide pour éviter d'avoir des
     * :machin.chose dans le contenu du mail
     */
    public function replaceLastVariables(string $text, Model $model): string
    {
        $allMatches = [];

        if (preg_match_all('/:model\.([_a-zA-Z0-9]+)?/', $text, $allMatches)) {
            foreach ($allMatches[1] as $key) {
                $value = $this->formatModelPropertyValue($model, $key);
                $text = str_replace(':model.'.$key, $value, $text);
            }
        }

        return $text;
    }

    public function getNotificationType(): NotificationType
    {
        return $this->notificationType;
    }

    public function getModel(): ?Content
    {
        return $this->model;
    }

    public function getUser(): ?User
    {
        return $this->user;
    }

    public function setUser(User $user): static
    {
        $this->user = $user;

        return $this;
    }

    public function getData(): ?array
    {
        return $this->data;
    }

    private function formatModelPropertyValue(Model $model, string $key): string
    {
        $return = $model->{$key} ?? '';
        $titleDateFormat = $this->data['extra']['titleDateFormat'] ?? self::DEFAULT_DATE_FORMAT;
        if (! empty($return) && in_array($key, self::DATE_FIELDS)) {
            $return = is_int($return) ? get_date($return)?->format($titleDateFormat) : $return->format($titleDateFormat);
        }

        return $return;
    }
}
