<?php

namespace Inside\Notify\Messages;

use Illuminate\Support\Str;
use libphonenumber\NumberParseException;
use libphonenumber\PhoneNumberFormat;
use libphonenumber\PhoneNumberUtil;
use libphonenumber\ShortNumberInfo;

class SmsMessage
{
    // The number of characters is limited to 11 for alphanumeric characters and 15 for numeric characters
    public const SENDER_LIMIT = 15;

    // Content of the message. If more than 160 characters long, will be sent as multiple text messages
    public const CONTENT_LIMIT = 160;

    /**
     * @var string
     */
    protected $sender;

    /** @var string */
    protected $recipient;

    /** @var string */
    protected $content;

    public function from(string $sender): self
    {
        $this->sender = Str::substr($sender, 0, self::SENDER_LIMIT);

        return $this;
    }

    /**
     * @param  string  $recipient
     * @return $this
     * @throws NumberParseException
     */
    public function to(string $recipient): self
    {
        $phoneUtil = PhoneNumberUtil::getInstance();
        $phoneNumber = $phoneUtil->parse($recipient, 'FR');
        if ($phoneUtil->isValidNumber($phoneNumber)) {
            $this->recipient = $phoneUtil->format($phoneNumber, PhoneNumberFormat::E164);
        }

        return $this;
    }

    /**
     * Message content
     *
     * @param  string  $content
     * @return $this
     */
    public function content(string $content): self
    {
        $this->content = Str::substr(
            inside_shortener_content(trim(strip_tags($content))),
            0,
            self::CONTENT_LIMIT
        );

        return $this;
    }

    /**
     * @return mixed
     */
    public function getContent()
    {
        return $this->content;
    }

    /**
     * @return string
     */
    public function getSender(): string
    {
        return $this->sender;
    }

    /**
     * @return mixed
     */
    public function getRecipient()
    {
        return $this->recipient;
    }

    public function toArray(): array
    {
        return [
            'sender' => $this->sender,
            'recipient' => $this->recipient,
            'content' => $this->content,
        ];
    }
}
