<?php

declare(strict_types=1);

namespace Inside\Notify\Messages;

use Illuminate\Support\Str;
use libphonenumber\NumberParseException;
use libphonenumber\PhoneNumberFormat;
use libphonenumber\PhoneNumberUtil;
use libphonenumber\ShortNumberInfo;

final class SendinBlueSmsMessage
{
    // The number of characters is limited to 11 for alphanumeric characters and 15 for numeric characters
    public const SENDER_LIMIT = 11;

    // Content of the message. If more than 160 characters long, will be sent as multiple text messages
    public const CONTENT_LIMIT = 160;

    public const MARKETING = 'marketing';

    public const TRANSACTIONAL = 'transactional';

    /**
     * @var string
     */
    protected $sender;

    /** @var string */
    protected $recipient;

    /** @var string */
    protected $content;

    /** @var string */
    protected $type = self::TRANSACTIONAL;

    /** @var string */
    protected $tag;

    /** @var string */
    protected $webUrl;

    public function from(string $sender): self
    {
        $this->sender = Str::substr($sender, 0, self::SENDER_LIMIT);

        return $this;
    }

    /**
     * @param string $recipient
     * @return $this
     * @throws NumberParseException
     */
    public function to(string $recipient): self
    {
        $phoneUtil = PhoneNumberUtil::getInstance();
        $phoneNumber = $phoneUtil->parse($recipient, 'FR');
        if ($phoneUtil->isValidNumber($phoneNumber)) {
            $this->recipient = $phoneUtil->format($phoneNumber, PhoneNumberFormat::E164);
        }

        return $this;
    }

    public function content(string $content): self
    {
        $this->content = Str::substr(trim(strip_tags($content)), 0, self::CONTENT_LIMIT);

        return $this;
    }

    public function type(string $type): self
    {
        $this->type = $type;

        return $this;
    }

    public function tag(string $tag): self
    {
        $this->tag = $tag;

        return $this;
    }

    public function webUrl(string $webUrl): self
    {
        $this->webUrl = $webUrl;

        return $this;
    }

    public function toArray(): array
    {
        return [
            'sender' => $this->sender,
            'recipient' => $this->recipient,
            'content' => $this->content,
            'type' => $this->type,
            'tag' => $this->tag,
            'webUrl' => $this->webUrl,
        ];
    }
}
