<?php

namespace Inside\Notify\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;
use Inside\Content\Models\Contents\Comments;
use Inside\Notify\Models\NotificationType;
use Inside\Reaction\Events\ReactionSubscribedEvent;
use Inside\Reaction\Models\Reaction;
use Inside\Slug\Facades\ContentUrlGenerator;

/**
 * Reaction subscribed listener
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class ReactionSubscribedListener extends BaseNotificationListener implements ShouldQueue
{
    /**
     * @param  ReactionSubscribedEvent  $event
     */
    public function handle(ReactionSubscribedEvent $event): void
    {
        $reaction = $event->reaction;
        $model = $reaction->reactionable;

        $types = NotificationType::where(
            [
                'event'  => ReactionSubscribedEvent::class,
                'model'  => $reaction->reactionable_type,
                'action' => $reaction->type,
            ]
        )->get();
        if ($types->isEmpty()) {
            $types = NotificationType::where(
                [
                    'event'  => ReactionSubscribedEvent::class,
                    'model'  => null,
                    'action' => $reaction->type,
                ]
            )->get();
        }

        foreach ($types as $type) {
            if (($reaction->type != $type->action)
                || ($type->model && $type->model != $reaction->reactionable_type)
            ) {
                continue;
            }

            $data = ['from' => $event->reaction->user_uuid];

            if ($reaction->reactionable_type == Comments::class) {
                // Special case for a comment
                $content = null;
                $parents = $model->getParentsIfExist();
                foreach ($parents as $parentContents) {
                    foreach ($parentContents as $parent) {
                        if ($parent->content_type != 'comments') {
                            $content = $parent;
                            break;
                        }
                    }
                }
                if ($content) {
                    $data['url'] = (ContentUrlGenerator::generateUrl($content) ?? '').'#&d/'.$model->uuid;
                }
            }

            if ($type->role) {
                $this->route = $type->roles->first();
                if (! $this->checkConditions($type, $model)) {
                    continue;
                }
                $this->notify($type, null, $model, $data, $type->roles->pluck('id'));
            } else {
                $type->subscribers->each(
                    function ($subscriber) use ($type, $model, $reaction, $data) {
                        if ($subscriber->uuid === $reaction->user_uuid) {
                            return;
                        }

                        $this->when = $this->getInterval($type, $model);
                        $this->route = $subscriber;

                        if (! $this->checkConditions($type, $model, $subscriber)
                            || ! $this->checkUniqueness(
                                $type,
                                $model,
                                $subscriber
                            )
                        ) {
                            return;
                        }

                        $this->notify($type, $subscriber, $model, $data);
                    }
                );
            }
        }
    }
}
