<?php

namespace Inside\Notify\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Str;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Content;
use Inside\Notify\Events\NotificationsManualsEvent;
use Inside\Notify\Jobs\SendManualNotifications;
use Inside\Notify\Models\NotificationType;
use Inside\Notify\Services\ManualNotificationsService;
use Inside\Notify\Services\NotificationLogService;
use Inside\Permission\Exodus\Actions\PermissionLogic\Role\CanReadContent;
use Inside\Permission\Models\Role;
use Inside\Permission\Services\PermissionService;

class NotificationManualsListener extends BaseNotificationListener implements ShouldQueue
{
    public function handle(NotificationsManualsEvent $event): void
    {
        if (empty($event->roleIds) || empty($event->channels)) {
            return; // No role means disabled
        }

        foreach ($event->channels as $channel) {
            $this->sendNotificationViaChannel(
                $channel,
                $event->contentType,
                $event->contentId,
                $event->roleIds
            );
        }
    }

    protected function sendNotificationViaChannel(string $channel, string $type, string $uuid, array $roleIds): void
    {
        $content = $this->getContent($type, $uuid);
        if (! $content->status) {
            return;
        }

        if (PermissionService::isSystemV2Enabled()) {
            $action = new CanReadContent();

            $roleIds = Role::query()
                ->whereIn('id', $roleIds)
                ->get()
                ->filter(fn (Role $role) => $action->handle($role, $content))
                ->pluck('id')
                ->toArray();
        }

        if (empty($roleIds)) {
            return;
        }

        if ($content->content_type === 'courses' && ! $content->published) {
            return;
        }

        $publishedAt = $this->getPublishedAt($content);
        $notificationType = $this->prepareAndCreate($channel, $type, $uuid);
        $jobId = null;

        if ($publishedAt && config('queue.default') === 'redis') {
            $queueKey = config('queue.connections.redis.queue');
            $job = new SendManualNotifications($channel, $roleIds, $content, $notificationType);
            $jobId = $job->getJobId();
            dispatch($job)->onQueue($queueKey)->delay($publishedAt);
        } else {
            $this->dispatchNotificationSending($channel, $roleIds, $content, $notificationType);
        }

        /**
         * @var NotificationLogService $notificationLogService
         */
        $notificationLogService = app()->make(NotificationLogService::class);
        $notificationLogService->logNotification(
            $channel,
            $roleIds,
            $content,
            $notificationType,
            $jobId,
            $publishedAt
        );
    }

    public function dispatchNotificationSending(
        string $channel,
        array $roleIds,
        Content $content,
        NotificationType $notificationType,
        ?string $queuedJobId = null
    ): void {
        $notificationType->roles()->sync($roleIds);

        /**
         * @var NotificationLogService $notificationLogService
         */
        $notificationLogService = app()->make(NotificationLogService::class);
        if ($queuedJobId) {
            $isProcessable = $notificationLogService->isProcessable($queuedJobId);
            if (! $isProcessable) {
                return;
            }
            $notificationLogService->markAsSent($queuedJobId);
        }

        if ($channel == 'web') {
            $this->subscribeUsersToNotification($roleIds, $notificationType);
            $this->route = $notificationType->roles->first();
            $this->notify($notificationType, null, $content, ['from' => $content->author], $roleIds);

            return;
        }

        /**
         * @var ManualNotificationsService $manualNotificationsService
         */
        $manualNotificationsService = app()->make(ManualNotificationsService::class);
        $emailNotificationType = $manualNotificationsService->notificationTypeForEmail();

        User::query()
            ->where('status', 1)
            ->whereHas('subscriptions', fn ($query) => $query->where('notification_type_id', $emailNotificationType->id))
            ->whereHas('roles', fn ($query) => $query->whereIn('role_id', $roleIds))
            ->where('langcode', $content->langcode)
            ->each(function (User $subscriber) use ($notificationType, $content) {
                $this->route = $subscriber;
                $this->notify($notificationType, null, $content, ['from' => $content->author]);
            });
    }

    protected function isEnhancedNotificationType(string $type): bool
    {
        if (! config('notify.manual_notifications.enhanced_notifications')) {
            return false;
        }

        return array_key_exists($type, config('notify.manual_notifications.enhanced_notifications_types_and_fields'));
    }

    protected function prepareAndCreate(string $channel, string $type, string $uuid): NotificationType
    {
        $customUrl = config('notify.manual_notifications.custom_url', false);
        $isEnhancedType = $this->isEnhancedNotificationType($type);
        $contentUrlField = config('notify.manual_notifications.contents_url_field', []);

        $url = null;
        if ($customUrl && is_callable($customUrl)) {
            $url = $customUrl($channel, $type, $uuid);
            if (! is_string($url)) {
                $url = null;
            }
        } elseif (! empty($contentUrlField) && array_key_exists($type, $contentUrlField)) {
            // $externals[$type] contains the field where to find the url
            $url = 'URLFIELD:'.$contentUrlField[$type];
        } elseif (Str::contains($type, 'document')) {
            // this look to be from a DMS
            $url = 'GED:'.str_replace($type, 'document', 'folder');
        }
        $fields = [
            'notification-data.from' => ['firstname', 'lastname'],
            'title',
            'content_type',
        ];
        if ($isEnhancedType || $type == 'external_notifications') {
            $typeFields = config('notify.manual_notifications.enhanced_notifications_types_and_fields')[$type]['fields'];
            $fields = array_merge($fields, $typeFields);
        }

        return NotificationType::create([
            'via' => $channel,
            'default' => true,
            'event' => NotificationsManualsEvent::class,
            'model' => type_to_class($type),
            'action' => 'manual',
            'multiple' => false,
            'profile' => false,
            'role' => true,
            'data' => [
                'title' => 'notifications.create.<content_type>.manual.title',
                'description' => 'notifications.create.<content_type>.manual.description',
                'text' => 'notifications.create.<content_type>.manual.text',
                'mail' => [
                    'subject' => 'notifications.create.<content_type>.manual.mail.subject',
                    /*'text'       => $isEnhancedType ? 'notifications.create.enhanced.<content_type>.manual.mail.content'
                        : 'notifications.create.<content_type>.manual.mail.content',*/
                    'text' => 'notifications.create.<content_type>.manual.mail.content',
                    'buttonText' => 'notifications.create.<content_type>.manual.mail.buttonText',
                ],
                'sms' => [
                    'content' => 'notifications.create.<content_type>.manual.sms.content',
                ],
                'push' => [
                    'subject' => 'notifications.create.<content_type>.manual.title',
                    'content' => 'notifications.create.<content_type>.manual.text',
                ],
                'view' => ($type == 'external_notifications')
                    ? 'externalNotificationEmail'
                    : ($isEnhancedType ? 'enhancedEmail' : 'email'),
                'fields' => $fields,
                'url' => $url,
            ],
        ]);
    }

    public function getContent(string $type, string $uuid): mixed
    {
        return call_user_func(type_to_class($type).'::findOrFail', $uuid);
    }

    protected function subscribeUsersToNotification(array $roleIds, NotificationType $notificationType): void
    {
        /**
         * @var ManualNotificationsService $manualNotificationsService
         */
        $manualNotificationsService = app()->make(ManualNotificationsService::class);
        $users = $manualNotificationsService
            ->webChanelSubscribers()
            ->pluck('uuid');

        $notificationType->subscribers()->syncWithoutDetaching($users);

        // Subscribe roles
        $roles = Role::whereIn('id', $roleIds)->pluck('id');
        $notificationType->roles()->syncWithoutDetaching($roles);
    }
}
