<?php

namespace Inside\Notify\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Log;
use Inside\Content\Events\CommentCreatedEvent;
use Inside\Content\Models\Contents\Comments;
use Inside\Notify\Models\NotificationType;
use Inside\Slug\Facades\ContentUrlGenerator;

/**
 * Comment created listener
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class CommentCreatedListener extends BaseNotificationListener implements ShouldQueue
{
    /**
     * @param CommentCreatedEvent $event
     */
    public function handle(CommentCreatedEvent $event): void
    {
        if ($event->comment->content_type != 'comments') {
            return;
        }
        Log::debug('CommentCreatedListener');

        // Comment created can trigger two notification type family
        // a comment attach to a content
        // a comment attach to another content
        $types = NotificationType::where('model', Comments::class)->where(
            function ($query) {
                $query->whereIn('action', ['contentCommented', 'commentCommented']);
            }
        )->get();
        $comment = $event->comment;
        $conditionTypes = [];
        foreach ($types as $type) {
            // Check comment obvious conditions

            // Let's check condition on parent content type
            if (! empty($type->condition)) {
                $conditions = explode('|', $type->condition);
                $conditionTypes = [];
                $newConditions = [];
                foreach ($conditions as $condition) {
                    [$name, $value] = explode(':', $condition, 2);
                    if ($name == 'parent_type') {
                        $conditionTypes = explode(',', $value);
                    } else {
                        $newConditions[] = $condition;
                    }
                }
                $type->condition = implode('|', $newConditions);
            }

            // Let's get the content that comment is attached to
            $parents = $comment->getParentsIfExist(
                count($conditionTypes) > 0 ? ['content_type' => $conditionTypes] : []
            );

            if (empty($parents) && $comment->parent == null) {
                continue; // It's too soon, pivot not set yet OR not correct content type
            }

            // Comment is only one level parent use it instead !
            if (! is_null($comment->parent)) {
                $parents['comments'] = [$comment->parent];
            }

            if (
                $type->action === 'commentCommented' &&
                (! array_key_exists('comments', $parents) || empty($parents['comments']))
            ) {
                continue; // This is not a comment's answer
            }

            $scrollComment = $comment;
            $model = $parent = null;

            switch ($type->action) {
                case 'commentCommented': // use parent comment, front can't scroll to comment responses
                    $parent = Arr::last($parents['comments']) ?? null;
                    $scrollComment = Arr::first($parents['comments']) ?? null;
                    foreach ($parents as $content_type => $contentParents) {
                        if ($content_type != 'comments' && ! empty($contentParents)) {
                            $model = Arr::first($contentParents);
                            break;
                        }
                    }
                    break;
                case 'contentCommented':
                    foreach ($parents as $content_type => $contentParents) {
                        if ($content_type != 'comments' && ! empty($contentParents)) {
                            $model = $parent = Arr::first($contentParents);
                            break;
                        }
                    }
                    break;
                default:
                    continue 2;
            }

            $this->when = $this->getInterval($type, $comment);
            $notificationData = [
                'from' => $comment->author,
                'comment' => $comment,
                'parentComment' => $scrollComment,
                'url' => (ContentUrlGenerator::generateUrl($model)).'#&d/'.$scrollComment->uuid,
            ];

            if ($type->role && $type->via !== 'email') {
                $this->route = $type->roles->first();
                if ($parent && $model && $this->checkConditions($type, $parent)) {
                    $this->notify($type, null, $model, $notificationData, $type->roles->pluck('id')->toArray());
                }
            } else {
                $type->subscribers->each(
                    function ($subscriber) use ($type, $comment, $parent, $model, $notificationData) {
                        if ($subscriber->uuid === $comment->author) {
                            return;
                        }

                        $this->route = $subscriber;

                        // Check additionnal conditions
                        // Now conditions are checked againts first parent
                        if (
                            $parent &&
                            $model &&
                            $this->checkConditions($type, $parent, $subscriber) &&
                            $this->checkUniqueness($type, $parent, $subscriber)
                        ) {
                            $roles = $type->role ? $type->roles->pluck('name')->toArray() : [];
                            $this->notify(
                                $type,
                                $subscriber,
                                $model,
                                $notificationData,
                                $roles
                            );
                        }
                    }
                );
            }
        }
    }
}
