<?php

namespace Inside\Notify\Listeners;

use Carbon\Carbon;
use Carbon\CarbonInterval;
use DateInterval;
use DateTimeInterface;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Str;
use Inside\Authentication\Models\User;
use Inside\Content\Exceptions\FieldSchemaNotFoundException;
use Inside\Content\Exceptions\ModelSchemaNotFoundException;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Content;
use Inside\Notify\Models\NotificationType;
use Inside\Notify\Notifications\InsideNotification;
use Inside\Notify\Notifications\MailNotification;
use Inside\Notify\Notifications\PushNotification;
use Inside\Notify\Notifications\SlackNotification;
use Inside\Notify\Notifications\SmsNotification;
use Inside\Notify\Notifications\WebNotification;
use Inside\Permission\Facades\Role as RoleService;
use Inside\Permission\Models\Role;
use Inside\Settings\Models\Setting;

class BaseNotificationListener
{
    public bool $deleteWhenMissingModels = true;

    protected ?Carbon $when = null;

    protected mixed $route = null;

    protected ?Role $role = null;

    /**
     * Notify user
     */
    public function notify(NotificationType $type, ?User $user, mixed $model = null, array $data = [], array $roles = [], ?string $theme = null): void
    {
        if ($user === null && empty($roles)) {
            $roles = $type->roles->pluck('id')->all();
        }

        if ($type->role && $model instanceof Content) {
            if (empty($roles)) {
                $roles = $type->roles->pluck('id')->all();
            }

            if ($type->via !== 'email' && $this->role) {
                $this->route = $this->role;
            }
        }

        if (is_null($user) && $this->route instanceof User) {
            $user = $this->route;
        }

        if ($type->via === 'email' && $model && $this->hasSpecificPermissions($model)) {
            if (is_null($user) || ! $user->hasAnyRole($this->getAllowedsRoles($model))) {
                return;
            }
        }

        if (is_null($user) && empty($roles)) {
            Log::warning('[notify] not possible :no recipient ...');

            return;
        }

        switch ($type->via) {
            case 'web':
                $notification = new WebNotification($type, $user, $model, $data, $roles);
                break;
            case 'email':
                /** @var User $user */
                if ($type->role && (empty($roles) || ! $user->hasAnyRole($roles))) {
                    return;
                }
                if ($this->role && ! $this->role->users->contains($user)) {
                    return;
                }
                $notification = new MailNotification($type, $user, $model, $data, $theme);
                // Route is email recipient
                $this->route = Notification::route('mail', $user->email);
                break;
            case 'sms':
                $notification = new SmsNotification($type, $user, $model, $data);
                break;
            case 'slack':
                $notification = new SlackNotification($type, $user, $model, $data);
                break;
            case 'push_desktop':
            case 'push_mobile':
                $notification = new PushNotification($type, $user, $model, $data);
                break;
            default:
                return;
        }

        $this->sendNotification($notification, $model);
    }

    /**
     * Last step: send the notification.
     *
     * @param InsideNotification $notification
     * @param mixed $model
     */
    public function sendNotification(InsideNotification $notification, $model): void
    {
        if ($this->route === null) {
            Log::warning('Can not send notification, no route set');

            return;
        }

        $publishedAt = $this->getPublishedAt($model);

        if ($publishedAt) {
            $this->route->notify($notification->delay($publishedAt));
        } else {
            $this->route->notify($notification);
        }
    }

    /**
     * Check if condition are filled
     *
     * @param NotificationType $type
     * @param mixed            $model
     * @param mixed            $user
     *
     * @return bool
     */
    public function checkConditions(NotificationType $type, $model = null, $user = null): bool
    {
        if (empty($type->condition)) {
            return true;
        }

        // Condition needs a model
        if (! empty($type->condition) && ! $model) {
            return false;
        }

        // User needs to be active !
        if ($user && ! $user->status) {
            return false;
        }

        $conditions = explode('|', $type->condition);

        foreach ($conditions as $condition) {
            $myModel = $model;
            $negativeCondition = false;

            // Each condition must be validate
            $condition = explode(':', $condition);
            if (Str::startsWith($condition[0], '*')) {
                continue; // Already checked by Updating listener and model has not changed yet
            }
            if (Str::startsWith($condition[0], '!')) {
                $condition[0] = substr($condition[0], 1);
                $negativeCondition = true;
            }
            $value = isset($condition[1]) ? $condition[1] : null;

            // cast value
            if ($value == 'true') {
                $value = true;
            } elseif ($value == 'false') {
                $value = false;
            } elseif ($value == 'null') {
                $value = null;
            }

            if ($user && $value == 'CURRENT_USER') {
                $value = $user->uuid;
            }

            $hasParent = explode('.', $condition[0]);
            if (count($hasParent) == 2) {
                if ($hasParent[0] == 'parent' && $myModel->pid) {
                    $myModel = call_user_func(get_class($myModel).'::find', $myModel->pid);
                } elseif (isset($myModel->{$hasParent[0]})) {
                    $myModel = $myModel->{$hasParent[0]};
                    $condition[0] = $hasParent[1];
                }
            }
            $field = null;
            try {
                $field = (object) Schema::getFieldOptions(class_to_type($myModel), $condition[0]);
            } catch (ModelSchemaNotFoundException $e) {
            } catch (FieldSchemaNotFoundException $e) {
            }
            $values = null;
            if (strpos($value, ',') !== false) {
                $values = explode(',', $value);
            }
            if ($negativeCondition) {
                // field is a reference
                if ($field && $field->type == 'reference' && isset($myModel->{Str::camel($condition[0])})) {
                    $references = $myModel->{Str::camel($condition[0])};

                    if (is_iterable($references)) {
                        $values = explode(',', $value);

                        // We check agains one reference but we don't have one ...
                        if (empty($references) && ! empty($values)) {
                            return false;
                        }

                        foreach ($references as $reference) {
                            if (in_array($reference->uuid, $values)) {
                                // Current reference uuid is not in accepted values
                                return false;
                            }
                        }
                    } elseif (! is_string($references) || ($references == $value)) {
                        return false;
                    }
                } elseif (isset($myModel->{$condition[0]}) || ($myModel->{$condition[0]} == $value)) {
                    return false;
                } elseif (! empty($value) && ! isset($myModel->{$condition[0]}) && empty($myModel->{$condition[0]})) {
                    // TODO check that one

                    return false;
                }
            } else {
                $found = $myModel->{$condition[0]};

                // field is a reference
                if ($field && $field->type == 'reference' && isset($myModel->{Str::camel($condition[0])})) {
                    $references = $myModel->{Str::camel($condition[0])};

                    if ($references instanceof Collection) {
                        $values = explode(',', $value);

                        if ($references->pluck('uuid')->intersect($values)->isEmpty()) {
                            return false;
                        }
                    } elseif (! is_string($references) || ($references != $value)) {
                        return false;
                    }
                } elseif (! isset($myModel->{$condition[0]}) || ($values === null && $myModel->{$condition[0]} != $value)
                    || (is_array($values) && ! in_array($myModel->{$condition[0]}, $values))
                ) {
                    return false;
                } elseif (empty($value) && isset($myModel->{$condition[0]}) && ! empty($myModel->{$condition[0]})) {
                    return false;
                }

                // check on language if needed
                if ($user && $type->language && $myModel->langcode != $user->langcode) {
                    return false;
                }
            }
        }

        // No condition failed!! yipi! it's ok
        return true;
    }

    /**
     * Check notification uniqueness
     *
     * @param NotificationType $type
     * @param mixed            $model
     * @param mixed            $user
     *
     * @return bool
     */
    public function checkUniqueness(NotificationType $type, $model, $user): bool
    {
        if (
            $type->multiple &&
            ! empty($model?->uuid_host) &&
            count(list_languages()) > 1 &&
            $model->langcode != $user->langcode
        ) {
            return false;
        }

        if ($type->multiple == true) {
            return true;
        }

        return ! DB::table('inside_notifications')->where(
            [
                'notification_type_id' => $type->id,
                'user_uuid'            => $user->uuid,
                'notifiable_type'      => $model ? get_class($model) : null,
                'notifiable_uuid'      => $model ? $model->getKey() : null,
            ]
        )->exists();
    }

    public function getInterval(NotificationType $type, mixed $model = null): ?Carbon
    {
        $when = Carbon::now();

        if (! isset($type->data['delay_from']) && ! isset($type->data['delay_interval'])) {
            return null;
        }

        if ($model && isset($type->data['delay_from']) && isset($model->{$type->data['delay_from']})) {
            $when = get_date($model->{$type->data['delay_from']});
        }

        if (isset($type->data['delay_interval']) && ! empty($type->data['delay_interval'])) {
            $delay = $type->data['delay_interval'];
            $invert = false;

            if ($delay[0] == '-') {
                $delay = trim($delay, '-');
                $invert = true;
            }

            $interval = new CarbonInterval($delay);
            $interval->invert = (int) $invert;
            $when?->add($interval);
        }

        return $when;
    }

    protected function hasSpecificPermissions(Content $content): bool
    {
        return DB::table('inside_permissions_schema')->join(
            'inside_roles_permissions_schema',
            function ($join) {
                $join->on(
                    'inside_roles_permissions_schema.permission_schema_id',
                    '=',
                    'inside_permissions_schema.id'
                );
            }
        )->where('inside_permissions_schema.authorizable_uuid', $content->uuid)->where(
            'inside_roles_permissions_schema.is_content_specific',
            true
        )->where('inside_permissions_schema.action', 'read')->exists();
    }

    protected function getAllowedsRoles(Content $content): Collection
    {
        return Role::all()->filter(function (Role $role) use ($content) {
            return $role->name === 'super_administrator' || RoleService::can('read', $content, $role->id);
        });
    }

    protected function getPublishedAt(?Content $model): ?Carbon
    {
        if ($this->when) {
            return $this->when;
        }

        $publishedAt = null;

        if (isset($model->published_at)) {
            $publishedAt = get_date($model->published_at);
        }

        return $publishedAt?->isFuture() ? $publishedAt : null;
    }
}
