<?php

namespace Inside\Notify\Jobs;

use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Inside\Authentication\Events\AuthenticationLoginEvent;
use Inside\Authentication\Models\User;
use Inside\Jobs\Job;
use Inside\Notify\Models\Notification;
use Inside\Notify\Models\NotificationType;

/**
 * Class CleanupUserNotifications
 */
class CleanupUserNotifications extends Job
{
    public function __construct(
        protected User $user
    ) {
    }

    /**
     * handle the job
     */
    public function handle(): void
    {
        $toDelete = [];
        if ($this->user->notifications === null) {
            return; // No notification
        }

        // Special welcome notification
        $welcomeNotificationType = NotificationType::where(
            [
                ['via', '=', 'web'],
                ['default', '=', '1'],
                ['event', '=', AuthenticationLoginEvent::class],
                ['type', '=', 'global'],
                ['condition', '=', 'uuid:CURRENT_USER'],
                ['multiple', '=', '0'],
            ]
        )->first();

        // Clean old notification, lost etc... for $this->user
        $this->user->notifications->each(
            function (Notification $notification) use (&$toDelete, $welcomeNotificationType) {
                try {
                    // Do not delete welcome notification or it will be triggered again
                    if ($notification->trigger === $welcomeNotificationType) {
                        return;
                    }

                    // Notifiable does not exist anymore
                    if (($notification->type !== 'system' && $notification->notifiable === null)
                        || // Wrong created_at
                        get_date((int) $notification->created_at) === null
                        || // Notification is global is older than 6 month
                        ($notification->type == 'global'
                            && get_date((int) $notification->created_at)->isBefore(
                                Carbon::now()->subMonths(config('notify.lifetimes.global', 6))
                            ))
                    ) {
                        $toDelete[] = $notification->id;
                    }
                } catch (\Throwable | \Exception $e) {
                    $toDelete[] = $notification->id;
                    Log::debug('[Notify::Cleanup](user) '.$e->getMessage());
                }
                if (count($toDelete) > 1000) {
                    DB::table('inside_notifications')->whereIn('id', $toDelete)->delete();
                    $toDelete = [];
                }
            }
        );
        DB::table('inside_notifications')->whereIn('id', $toDelete)->delete();
    }
}
