<?php

namespace Inside\Notify\Http\Middlewares;

use Closure;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inside\Content\Facades\Schema as InsideSchema;
use Inside\Notify\Facades\OneSignal;
use Inside\Notify\Facades\SMS;
use Inside\Permission\Facades\Permission;
use Symfony\Component\HttpFoundation\Response;

/**
 * Class ContentRoleFormMiddleware
 */
class ContentRoleFormMiddleware
{
    /**
     * Handle an incoming request.
     *
     * @param  Request  $request
     * @param  Closure  $next
     *
     * @return mixed
     */
    public function handle(Request $request, Closure $next)
    {
        $path = $request->path();

        $response = $next($request);

        if (strpos($path, 'api/v1/form/') === false || $request->method() !== 'GET') {
            return $response; // Wrong endpoint
        }

        $config = config('notify.manual_notifications', ['enabled' => false]);
        if ($config['enabled'] !== true) {
            return $response;
        }

        $type = $request->route()[2]['type'];

        if (! empty($config['enabled_types']) && ! in_array($type, $config['enabled_types'])) {
            return $response;
        }

        if (! empty($config['disabled_types']) && in_array($type, $config['disabled_types'])) {
            return $response;
        }

        $data = json_decode_response($response);
        if ($response instanceof Response && $response->getStatusCode() != 200) {
            return $response;
        }

        $groupRolesLabels = [];
        $groupRolesWarningMessage = [];
        foreach (list_languages() as $locale) {
            $groupRolesLabels[$locale] = __('notifications.forms.notification.label', [], $locale);
            $groupRolesWarningMessage[$locale] = __('notifications.forms.notification.warning_message', [], $locale);
        }

        $fields = [];
        $fields[] = Permission::getRolePickerFieldForFrontForm(
            'notifications_roles',
            'notifications.forms.title.roles.label',
            null,
            [],
            51,
            true
        );

        $fields[] = $this->prepareNotificationTransporterTypesFieldForFront($type);

        $config = [
            'id' => 'group_manual_notification',
            'weight' => 50,
            'type' => 'fieldset',
            'classes' => '',
            'label' => $groupRolesLabels,
            'fields' => $fields,
        ];

        if ($type === 'courses') {
            $config['warning_message'] = $groupRolesWarningMessage;
        }

        $data['data'][] = $config;

        set_response($response, $data);

        return $response;
    }

    /**
     * @param  string  $type
     * @return array
     * @throws Exception
     */
    protected function prepareNotificationTransporterTypesFieldForFront(string $type): array
    {
        $modelOptions = InsideSchema::getModelOptions($type);
        $default = [];
        if (array_key_exists('notifications', $modelOptions) && $modelOptions['notifications'] !== 'disabled') {
            switch ($modelOptions['notifications']) {
                case 'web':
                    $default = ['web'];
                    break;
                case 'email':
                    $default = ['email'];
                    break;
                case 'both':
                    $default = ['web', 'email'];
                    break;
            }
        }

        $labels = [];
        $allowedValues = [
            'vias' => [],
        ];
        $disabledValues = [];
        $values = [
            'web',
            'email',
        ];
        if (SMS::isAvailable()) {
            $values[] = 'sms';
        }
        if (OneSignal::isAvailable()) {
            if (OneSignal::isMobileActivated()) {
                $values[] = 'push_mobile';
            }

            if (OneSignal::isDesktopActivated()) {
                $values[] = 'push_desktop';
            }
        }
        foreach (list_languages() as $locale) {
            $labels[$locale] = __('notifications.forms.title.via.label', [], $locale);
            $translations = [];
            $disabledValuesLabels = [];
            foreach ($values as $value) {
                $qualifiers = [];
                if ($value == 'sms' && null !== ($quota = SMS::getQuota()) && $quota > 0) {
                    $availableSms = $quota - SMS::getSmsCountForCurrentMonth();
                    if ($availableSms < 0) {
                        $availableSms = 0;
                    }
                    $availableSmsInPercent = $availableSms / $quota;
                    $color = setting('_theme', 'colorValid');
                    if ($availableSmsInPercent <= 0.1) {
                        $color = setting('_theme', 'colorDanger');
                    } elseif ($availableSmsInPercent < 0.3) {
                        $color = setting('_theme', 'colorWarn');
                    }
                    $qualifiers[] = [
                        'title' => trans_choice(
                            is_maecia_admin() ? 'notifications.sms.maecia_remaining' : 'notifications.sms.remaining',
                            $availableSms,
                            [
                                'count' => $availableSms,
                                'remaining' =>  is_maecia_admin() ? SMS::getFormattedRemainingCredits() : null,
                            ]
                        ),
                        'color' => $color,
                        'position' => 'after',
                    ];
                }

                $translations[] = [
                    'title' => __('notifications.via.'.$value, [], $locale),
                    'uuid' => $value,
                    'children' => [],
                    'qualifiers' => $qualifiers,
                ];
                if ($value == 'sms' && SMS::overQuota()) {
                    $disabledValuesLabels[$value] = __('notifications.sms.over_quota', [], $locale);
                }
            }
            $disabledValues[$locale] = $disabledValuesLabels;
            $allowedValues['vias'][$locale] = $translations;
        }

        return [
            'name' => 'notifications_vias',
            'options' => [
                'cardinality' => -1,
                'categorizable' => false,
                'disabled_values' => $disabledValues,
                'allowed_values' => $allowedValues,
                'default' => $default,
                'title' => $labels,
                'widget' => 'select',
                'front_field_config' => [
                    'multiple' => true,
                ],
            ],
            'type' => 'select',
            'weight' => 52,
        ];
    }
}
