<?php

namespace Inside\Notify\Console;

use Illuminate\Console\Command;
use Illuminate\Contracts\Support\Arrayable;
use Inside\Content\Models\Content;
use Inside\Content\Models\Field;
use Inside\Notify\Console\Helper\MarkdownTable;
use Inside\Notify\Models\NotificationType;

/**
 * Class DumpCommand
 */
class DumpCommand extends Command
{
    /**
     * @var string
     */
    protected $name = 'notify:dump';

    /**
     * @var string
     */
    protected $signature = 'notify:dump {--markdown}';

    /**
     * @var string
     */
    protected $description = 'Dump actual notification';

    /**
     * Handle index command
     *
     * @return void
     * @throws \Exception
     */
    public function handle(): void
    {
        $markdown = $this->option('markdown');
        foreach (['email', 'web'] as $type) {
            $this->info($type);
            $notificationTypes = NotificationType::where('via', $type)->with('subscribers')->get();
            $values = $notificationTypes->transform(
                function ($notificationType) {
                    return [
                        $notificationType->id,
                        ! empty($notificationType->model)
                            ? class_to_type($notificationType->model)
                            : class_to_type($notificationType->event),
                        $this->cleanCondition($notificationType->condition, $notificationType->model),
                        $this->getAction($notificationType->action),
                        ($notificationType->default ? 'Oui' : 'Non'),
                        ($notificationType->profile ? 'Oui' : 'Non'),
                        $this->getUsers($notificationType),
                    ];
                }
            );
            if (! $markdown) {
                $this->table(
                    ['ID', 'Type', 'Condition', 'Action', 'Par défaut', 'Dans le profile', 'Qui'],
                    $values
                );
            } else {
                $this->markdownTable(
                    ['ID', 'Type', 'Condition', 'Action', 'Par défaut', 'Dans le profile', 'Qui'],
                    $values
                );
            }
            $this->line('');
        }
    }

    /**
     * get Human reading conditions
     *
     * @param string $conditions
     * @param Content $model
     * @return string
     */
    protected function cleanCondition(string $conditions, Content $model): string
    {
        if (empty($conditions)) {
            return 'Aucune';
        }

        return collect(explode('|', $conditions))->transform(
            function ($condition) use ($model) {
                $condition = explode(':', $condition);
                $value = $condition[1] ?? null;

                if ($value == 'CURRENT_USER') {
                    return $condition[0].' est [Utilisateur Connecté]';
                }

                /** @var Field|null $field */
                $field = Field::where(
                    [
                        'model' => $model,
                        'name'  => $condition[0],
                    ]
                )->first();
                if ($field && $field->type == 'reference') {
                    $values = explode(',', (string) $value);
                    if (count($values) > 1) {
                        return $condition[0].' est dans ["'.collect($values)->transform(
                            function ($value) use ($condition) {
                                $query = call_user_func(
                                    type_to_class($condition[0] == 'author' ? 'users' : $condition[0]).'::query'
                                );
                                $model = $query->find($value);
                                if ($model) {
                                    return $model->title ?? ('<'.$model->email.'>');
                                }

                                return ' ### contenu effacé ### ';
                            }
                        )->implode('", "').'"]';
                    } else {
                        $query = call_user_func(
                            type_to_class($condition[0] == 'author' ? 'users' : $condition[0]).'::query'
                        );
                        $model = $query->find($value);
                        if ($model) {
                            return $condition[0].' est "'.($model->title ?? ('<'.$model->email.'>').'"');
                        }

                        return $condition[0].' est ### contenu effacé ### ';
                    }
                }

                return $condition[0].' est '.$condition[1];
            }
        )->implode(' / ');
    }

    /**
     * Get subscribers emails
     *
     * @param NotificationType $notificationType
     * @return string
     */
    protected function getUsers(NotificationType $notificationType): string
    {
        if ($notificationType->subscribers()->where('status', 1)->count() > 0) {
            return '<'.$notificationType->subscribers->where('status', 1)->implode('email', '>, <').'>';
        } else {
            return 'Personne';
        }
    }

    /**
     * Get action has human reading info
     *
     * @param string $action
     * @return string
     */
    protected function getAction(string $action): string
    {
        switch ($action) {
            case 'create':
                return 'Création';
            case 'createOrUpdate':
                return 'Création ou Modification';
            case 'updating':
                return 'Modification avec certaines conditions';
            case 'report':
                return 'Report';
            case 'edit':
                return 'Edition';
            case 'submit':
                return 'Envoi de formulaire';
            case 'follow':
                return 'Cliquer sur "Suivre"';
            case 'like':
                return 'Cliquer sur "J\'aime"';
            case 'join':
                return 'Rejoindre';
            case 'login':
                return 'S\'identifier';
            case 'leveled_up':
                return 'A pris un niveau';
            case 'new_badge':
                return 'Gain d\'un badge';
            case 'accept':
                return 'Proposition acceptée';
            case 'decline':
                return 'Proposition refusée';
            default:
                return $action;
        }
    }

    /**
     * @param array $headers
     * @param mixed $rows
     * @param string $tableStyle
     * @param array $columnStyles
     * @return void
     */
    protected function markdownTable($headers, $rows, string $tableStyle = 'default', array $columnStyles = []): void
    {
        $table = new MarkdownTable($this->output);

        if ($rows instanceof Arrayable) {
            $rows = $rows->toArray();
        }

        $table->setHeaders((array) $headers)->setRows($rows)->setStyle($tableStyle);

        foreach ($columnStyles as $columnIndex => $columnStyle) {
            $table->setColumnStyle($columnIndex, $columnStyle);
        }

        $table->render();
    }
}
