<?php

namespace Inside\Notify\Channels;

use Closure;
use Exception;
use Illuminate\Contracts\Mail\Mailable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Channels\MailChannel as BaseMailChannel;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Facades\Log;
use Inside\Notify\Notifications\MailNotification;
use Swift_Mime_ContentEncoder_PlainContentEncoder;

/**
 * Mail channel.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class MailChannel extends BaseMailChannel
{
    /**
     * @inheritDoc
     */
    public function send($notifiable, Notification $notification)
    {
        /** @var MailNotification $notification */
        $message = $notification->toMail($notifiable);

        if (! $notifiable->routeNotificationFor('mail') && ! $message instanceof Mailable) {
            return;
        }

        if ($message instanceof Mailable) {
            $message->send($this->mailer);

            return;
        }
        Log::debug('mail channel ['.json_encode($notifiable->routeNotificationFor('mail')).']');

        try {
            $this->mailer->send(
                $this->buildView($message),
                array_merge($message->data(), $this->additionalMessageData($notification)),
                $this->messageBuilder($notifiable, $notification, $message)
            );
        } catch (Exception $exception) {
            // an error occurred when trying to send a mail, don't block execution of other tasks.
            Log::error('[MailChannel] failed to send message ['.$exception->getMessage().']');
        }
    }

    /**
     * @param  mixed  $notifiable
     * @param  Notification  $notification
     * @param  MailMessage  $message
     * @return Closure
     */
    protected function messageBuilder($notifiable, $notification, $message): Closure
    {
        if (! config('notify.override_message_builder')) {
            return parent::messageBuilder($notifiable, $notification, $message);
        }

        return function ($mailMessage) use ($notifiable, $notification, $message) {
            $this->buildMessage($mailMessage, $notifiable, $notification, $message);
        };
    }

    /**
     * @param  MailMessage  $message
     * @return array|string
     */
    protected function buildView($message): array|string
    {
        if ($message->view) {
            return $message->view;
        }

        if (property_exists($message, 'theme') && ! is_null($message->theme)) {
            $this->markdown->theme($message->theme);
        }

        return [
            'html' => $this->markdown->render((string) $message->markdown, $message->data()),
            'text' => $this->markdown->renderText((string) $message->markdown, $message->data()),
        ];
    }

    /**
     * {@inerhitDoc}
     */
    protected function additionalMessageData($notification): array
    {
        return [
            '__laravel_notification_id' => $notification->id,
            '__laravel_notification' => get_class($notification),
            '__laravel_notification_queued' => in_array(
                ShouldQueue::class,
                (array) class_implements($notification)
            ),
        ];
    }
}
