<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Schema;

class CreateNotificationsViewsTable extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        if (! Schema::hasTable('inside_notifications_users_views')) {
            Schema::create(
                'inside_notifications_users_views',
                function (Blueprint $table) {
                    $table->increments('id')->collate('utf8');
                    $table->integer('notification_id')->collate('utf8')->unsigned();
                    $table->uuid('user_uuid')->collate('utf8');
                    $table->timestamp('view_at')->nullable();
                    $table->timestamp('read_at')->nullable();

                    $table->foreign('notification_id')->references('id')->on('inside_notifications')->onDelete('cascade');
                    $table->foreign('user_uuid')->references('uuid')->on('inside_users')->onDelete('cascade');

                    $table->unique(['user_uuid', 'notification_id'], 'unique_notifications_users_views');
                    $table->index(['user_uuid']);
                }
            );
        }

        if (Schema::hasColumn('inside_notifications', 'view_at') && Schema::hasColumn('inside_notifications', 'read_at')) {
            Inside\Notify\Models\Notification::whereNotNull('user_uuid')
                ->whereNotNull('view_at')
                ->get()
                ->each(
                    function ($notification) {
                        Inside\Notify\Models\NotificationUserView::UpdateOrCreate(
                            [
                                'notification_id' => $notification->id,
                                'user_uuid' => $notification->user_uuid,
                            ],
                            [
                                'view_at' => is_null($notification->view_at) ? null : Carbon::createFromTimeString($notification->view_at),
                                'read_at' => is_null($notification->read_at) ? null : Carbon::createFromTimeString($notification->read_at),
                            ]
                        );
                    }
                );

            Schema::table(
                'inside_notifications',
                function (Blueprint $table) {
                    $table->dropColumn(['view_at', 'read_at']);
                }
            );
        }
    }

    public function down(): void
    {
        if (! Schema::hasColumn('inside_notifications', 'view_at') || ! Schema::hasColumn('inside_notifications', 'read_at')) {
            Schema::table(
                'inside_notifications',
                function (Blueprint $table) {
                    $table->timestamp('view_at')->nullable();
                    $table->timestamp('read_at')->nullable();
                }
            );
        }
        Schema::dropIfExists('inside_notifications_users_views');
    }
}
