<?php

namespace Inside\Menu\Services;

use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Content;
use Inside\Menu\Contracts\Menu as MenuContract;
use Inside\Menu\Models\MenuLink;
use Inside\Permission\Exodus\Actions\SortPrivileges\SortByDepth;
use Inside\Permission\Exodus\Dto\Privileges\MenuPrivilegeDto;
use Inside\Permission\Exodus\Exceptions\UnauthorizedPermissionException;
use Inside\Permission\Facades\Permission;

class MenuService implements MenuContract
{
    private bool $unlimited = false;

    private array $allowed = [];

    public function __construct(
        protected ?User $user = null
    ) {
        /** @var User $user */
        $user = Auth::user();

        $this->user = $user;
        $this->unlimited = php_sapi_name() === 'cli' || $user->isSuperAdmin() || $user->hasBackofficeAccessTo('menu');

        if (! $this->unlimited) {
            $this->allowed = $user
                ->getAuthorizedMenuPrivileges()
                ->map(fn (MenuPrivilegeDto $dto) => $dto->toArray())
                ->pluck('index.uuid')
                ->toArray();
        }
    }

    /**
     * Create a new menu link
     */
    public function create(string $menuName, array $data): MenuLink
    {
        $this->uploadImage($data);

        $data['weight'] = 0;

        $link = call_user_func(menu_type_to_class($menuName).'::query')
            ->where('langcode', $data['langcode'])
            ->whereNull('parent_uuid')
            ->orderByDesc('weight')
            ->first();

        if ($link) {
            $data['weight'] = $link->weight + 1;
        }

        if (array_key_exists('link', $data) && is_array($data['link'])) {
            $data['link'] = sprintf('%s|%s', $data['link']['content_type'], $data['link']['uuid']);
        }

        return call_user_func(menu_type_to_class($menuName).'::create', $data);
    }

    /**
     * Move a menu link (change parent and / or weight)
     */
    public function move(string $menuName, string $uuid, array $data): MenuLink
    {
        $data['parent_uuid'] = $data['parent_uuid']['parent_uuid'];

        $menuLink = call_user_func(menu_type_to_class($menuName).'::find', $uuid);

        $siblings = call_user_func(menu_type_to_class($menuName).'::query')
            ->where('parent_uuid', $data['parent_uuid'])
            ->where('weight', '>=', $data['weight'])
            ->where('langcode', $menuLink->langcode)
            ->where('uuid', '<>', $uuid)
            ->get();

        $weight = $data['weight'];

        foreach ($siblings as $sibling) {
            $sibling->weight = ++$weight;
            $sibling->save();
        }

        $menuLink->weight = $data['weight'];
        $menuLink->parent_uuid = $data['parent_uuid'];
        $menuLink->save();

        return $menuLink;
    }

    /**
     * Update a menu link
     */
    public function update(string $menuName, string $uuid, array $data): MenuLink
    {
        $this->uploadImage($data);

        if (array_key_exists('link', $data) && is_array($data['link'])) {
            $data['link'] = sprintf('%s|%s', $data['link']['content_type'], $data['link']['uuid']);
        }

        $menu = call_user_func(menu_type_to_class($menuName).'::findOrFail', $uuid);
        $menu->update($data);

        return $menu;
    }

    /**
     * Delete a menu link
     */
    public function delete(string $menuName, string $uuid): void
    {
        $menuLink = call_user_func(menu_type_to_class($menuName).'::findOrFail', $uuid);

        $children = Permission::isSystemV2Enabled()
            ? menu_type_to_class($menuName)::query()->where('parent_uuid', $uuid)->get()->toArray()
            : $this->getMenuTree($menuName, $menuLink->langcode, $uuid, true);

        foreach ($children as $child) {
            DB::table((string) menu_type_to_table($menuName))->where('uuid', $child['uuid'])->update([
                'parent_uuid' => $menuLink->parent_uuid,
                'weight' => $menuLink->weight,
            ]);
        }

        $menuLink->delete();
    }

    /**
     * Get a given menu link listing
     */
    public function getMenuLinks(string $menuName, string $langcode, bool $admin = false, string $parent = null, bool $getChildren = true): array
    {
        return $this->getMenuTree($menuName, $langcode, $parent, $admin, $getChildren);
    }

    /**
     * Get the menu tree
     */
    protected function getMenuTree(string $menuName, string $langcode, string $parent = null, bool $admin = false, bool $getChildren = true): array
    {
        return (Permission::isSystemV2Enabled())
            ? $this->getMenuTreeV2($menuName, $langcode, $parent, $admin, $getChildren)
            : $this->getMenuTreeV1($menuName, $langcode, $parent, $admin, $getChildren);
    }

    protected function getMenuTreeV1(string $menuName, string $langcode, string $parent = null, bool $admin = false, bool $getChildren = true): array
    {
        $query = call_user_func(menu_type_to_class($menuName).'::query');

        if (! $admin) {
            $query->where('status', 1);
        }

        $menuLinks = $query->where('langcode', $langcode)
            ->where('parent_uuid', $parent)
            ->orderBy('weight', 'asc')
            ->get()
            ->toArray();

        foreach ($menuLinks as $key => &$menuLink) {
            if (! $this->readable($menuName, $menuLink)) {
                array_splice($menuLinks, $key, 1);
                continue;
            }

            $this->transformLink($menuLink);

            if ($getChildren) {
                $menuLink['children'] = $this->getMenuTree($menuName, $langcode, $menuLink['uuid'], $admin);
            }
        }

        return $menuLinks;
    }

    protected function getMenuTreeV2(string $menuName, string $langcode, string $parent = null, bool $admin = false, bool $getChildren = true): array
    {
        $menus = menu_type_to_class($menuName)::query()
            ->when(! $admin, fn ($query) => $query->where('status', 1))
            ->when(! $this->unlimited, fn ($query) => $query->whereIn('uuid', $this->allowed))
            ->where('langcode', $langcode)
            ->orderBy('weight', 'asc')
            ->get();

        $allowed = $menus->mapWithKeys(function (MenuLink $menu) use ($menus) {
            $key = $menu->uuid;
            $data = $menu->toArray();

            $this->transformLink($data);

            return match (true) {
                // If the menu does not have link, we allow it (example: separator / image)
                ! is_array($data['link']) => [$key => $data],

                // If the menu has a link, we check if the user has the permission to read it
                (bool) data_get($data, 'link.title') => [$key => $data],

                // If not, we allow it if it has children but we remove the link
                $menus->contains('parent_uuid', $key) => [$key => array_merge($data, ['link' => null])],

                // If not, we remove it
                default => null,
            };
        });

        return (new SortByDepth())->execute($allowed->filter(), 'parent_uuid')->values()->toArray();
    }

    /**
     * Check if a menu link is readable
     */
    protected function readable(string $menuName, array $menuLink): bool
    {
        if (php_sapi_name() === 'cli') {
            return true;
        }

        if (is_null($this->user)) {
            return false;
        }

        if (collect(config('menu.public'))->contains($menuName)) {
            return true;
        }

        if ($this->user->hasAnyRole('super_administrator')) {
            return true;
        }

        foreach ($this->user->roles as $role) {
            if ($role->hasAccessToBackoffice('menu')) {
                return true;
            }
        }

        return DB::table('inside_permissions')
            ->where('inside_permissions.action', 'read')
            ->where('inside_permissions.type', menu_type_to_class($menuName))
            ->when($menuLink['uuid'], function ($query) use ($menuLink) {
                $query->where('inside_permissions.uuid', $menuLink['uuid']);
            }, function ($query) {
                $query->whereNull('inside_permissions.uuid');
            })
            ->whereIn('inside_permissions.role_id', $this->user->getRoleIds())
            ->exists();
    }

    /**
     * Handle image upload
     */
    public function uploadImage(array &$data): void
    {
        if (! array_key_exists('image', $data)) {
            return;
        }
        if (! Storage::disk('local')->exists('menus')) {
            Storage::disk('local')->makeDirectory('menus');
        }

        /** @var array $pathInfo */
        $pathInfo = pathinfo($data['image']->getFilename());
        $path = sprintf('menus/%s.%s', $pathInfo['filename'], $pathInfo['extension']);
        $i = 1;

        while (Storage::disk('local')->exists($path)) {
            $path = sprintf('menus/%s-%d.%s', $pathInfo['filename'], $i++, $pathInfo['extension']);
        }

        $pathInfo = pathinfo($path);
        $directory = str_replace(cms_base_path('').'/', '', Storage::disk('local')->path($pathInfo['dirname']));
        $data['image']->move($directory, $pathInfo['basename']);

        $originalDirectory = str_replace(cms_base_path('').'/'.config('app.app_storage_path', 'storage/app').'/', '', $data['image']->getPath());

        if (preg_match('#((chunks|fakes)/[^/]+)#', $originalDirectory, $matches) === 1
            && empty(Storage::disk('local')->listContents($matches[1]))
        ) {
            Storage::disk('local')->deleteDir($matches[1]);
        }

        $data['image'] = $path;
    }

    /**
     * Format the link attribute of a menu link
     */
    public function transformLink(array &$menuLink): void
    {
        (Permission::isSystemV2Enabled())
            ? $this->transformLinkV2($menuLink)
            : $this->transformLinkV1($menuLink);
    }

    public function transformLinkV1(array &$menuLink): void
    {
        if (! Str::contains($menuLink['link'], '|')) {
            return;
        }

        try {
            [$contentType, $uuid] = explode('|', $menuLink['link']);
            $content = call_user_func(type_to_class($contentType).'::find', $uuid);

            $menuLink['link'] = [
                'content_type' => $contentType,
                'uuid' => $uuid,
                'title' => $content->title,
                'slug' => $content->slug[0] ?? null,
            ];
        } catch (Exception) {
            $menuLink['link'] = null;
        }
    }

    public function transformLinkV2(array &$menuLink): void
    {
        if (! isset($menuLink['children'])) {
            $menuLink['children'] = [];
        }

        if (! Str::contains($menuLink['link'], '|')) {
            return;
        }

        [$type, $uuid] = explode('|', $menuLink['link'] ?? '');

        $content = call_user_func(type_to_class($type).'::find', $uuid);

        $menuLink['link'] = [
            'content_type' => $type,
            'uuid' => $uuid,
            'title' => $content?->title,
            'slug' => data_get($content, 'slug.0'),
        ];
    }
}
