<?php

use Doctrine\DBAL\Schema\Index;
use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Inside\Menu\Models\MenuLink;
use Inside\Permission\Models\PermissionSchema;
use Inside\Support\Str;

final class AlterMenuToUseUuid extends Migration
{
    public function up(): void
    {
        $menus = [
            'header',
            'footer',
        ];

        foreach ($menus as $menu) {
            $tableName = menu_type_to_table($menu);

            Schema::table($tableName, function (Blueprint $table) {
                $table->uuid('uuid')->nullable()->default(null);
                $table->uuid('parent_uuid')->nullable()->default(null);
            });

            DB::table($tableName)->get()->each(fn ($link) => DB::update(
                query: "UPDATE {$tableName} SET uuid = ? WHERE id = ?",
                bindings: [(string) Str::uuid(), $link->id]
            ));

            $mapping = collect(DB::select("SELECT id, uuid FROM {$tableName}"))
                ->mapWithKeys(fn (Object $link) => [$link->uuid => $link->id]);

            DB::table($tableName)->whereNotNull('parent_id')->get()->each(fn ($link) => DB::update(
                query: "UPDATE {$tableName} SET parent_uuid = ? WHERE id = ?",
                bindings: [$mapping->search($link->parent_id), $link->id]
            ));

            DB::table('inside_permissions')
                ->whereNotNull('uuid')
                ->where('type', menu_type_to_class($menu))
                ->whereIn('uuid', $mapping->values())
                ->get()
                ->each(fn ($link) => DB::update(
                    query: 'UPDATE inside_permissions SET uuid = ? WHERE id = ?',
                    bindings: [$mapping->search($link->uuid), $link->id]
                ));

            PermissionSchema::query()
                ->where('authorizable_type', menu_type_to_class($menu))
                ->whereIn('authorizable_uuid', $mapping->values()->toArray())
                ->get()
                ->each(function (PermissionSchema $permission) use ($mapping) {
                    $id = (int) $permission->authorizable_uuid;
                    $permission->authorizable_uuid = $mapping->search($id);
                    $permission->save();
                });

            $indexes = collect(Schema::getConnection()->getDoctrineSchemaManager()->listTableIndexes($tableName))
                ->reject(fn (Index $index) => $index->isPrimary());

            Schema::table($tableName, function (Blueprint $table) use ($indexes, $tableName) {
                foreach ($indexes as $index) {
                    $table->dropForeign($index->getName());
                }

                $table->dropColumn('parent_id');
                $table->dropColumn('id');

                $table->primary('uuid');
                $table->foreign('parent_uuid')->references('uuid')->on($tableName)->onDelete('set null');
            });
        }
    }
}
