<?php

namespace Inside\Layout\Repositories;

use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Inside\Content\Facades\Schema;
use Inside\Layout\Exceptions\NeedRedirectToLayout;
use Inside\Layout\Models\Layout;
use Inside\Layout\Models\LayoutPath;
use Inside\Layout\Repositories\Contracts\LayoutRepositoryInterface;
use Inside\Layout\Validators\LayoutValidator;
use Inside\Slug\Models\Slug;

class LayoutRepository implements LayoutRepositoryInterface
{
    /**
     * @param LayoutPath $layoutPath
     * @param string $langcode
     * @return string|null
     */
    protected function getCorrectPath(LayoutPath $layoutPath, string $langcode): ?string
    {
        $layout = $layoutPath->layout;
        $paths = $layout->paths->where('langcode', $langcode);
        if ($paths->isEmpty()) {
            return null;
        }

        return $paths->first()->path;
    }

    /**
     * @param Slug $slug
     * @param string $langcode
     * @return Slug|null
     */
    protected function getCorrectSlug(Slug $slug, string $langcode): ?Slug
    {
        $type = str_replace('inside_content_', '', $slug->type ?? '');

        if (Schema::hasContentType($type)) {
            try {
                $current = call_user_func(type_to_class($type).'::findOrFail', $slug->uuid);
                $query = call_user_func(type_to_class($type).'::query');
                $translated = $query->where(
                    [
                        ['uuid_host', '=', $current->uuid_host],
                        ['langcode', '=', $langcode],
                    ]
                )->firstOrFail();
                $slug = Slug::where(
                    [
                        ['uuid', '=', $translated->uuid],
                        ['type', '=', $slug->type],
                    ]
                )->firstOrFail();
            } catch (ModelNotFoundException $e) {
                return null;
            }

            return $slug;
        }

        return null;
    }

    /**
     * @param string $path
     * @param string $langcode
     * @return mixed
     */
    public function current(string $path, string $langcode = '')
    {
        $me = Auth::user();
        if (! $me) {
            return null;
        }

        if (empty($langcode)) {
            $langcode = $me->langcode;
        }
        $exploded = explode('/', $path);

        $this->manageEdition($exploded, $path, $langcode);

        $layoutPath = LayoutPath::where(
            [
                ['path', $path],
                ['langcode', $langcode],
            ]
        )->first();

        if (! $layoutPath) {
            $slug = Slug::where('slug', '=', $exploded[0])->first();

            if ($slug) {
                if ($slug->langcode != $langcode) {
                    $this->validateSlug($slug, $langcode);
                } else {
                    $layoutPath = $this->checkCondition($slug, $path);
                    if (! $this->checkCondition($slug, $path) && ! isset($exploded[1])) {
                        $layoutPath = LayoutPath::where('path', '=', $slug->type)->first();
                    }
                }
            }
        }

        if ($layoutPath) {
            return $layoutPath->layout()->first();
        }

        return null;
    }

    /**
     * @param array $exploded
     * @param string $path
     * @param string $langcode
     * @return void
     */
    private function manageEdition(array $exploded, string $path, string $langcode): void
    {
        if (count($exploded) == 3 && $exploded[0] == 'edit') {
            if ($exploded[1] == 'users') {
                throw new NeedRedirectToLayout($path, $langcode);
            }

            $query = call_user_func('\\Inside\\Content\\Models\\Contents\\'.Str::studly($exploded[1]).'::query');
            $current = $query->find($exploded[2]);
            if ($current->langcode != $langcode) {
                $query = call_user_func('\\Inside\\Content\\Models\\Contents\\'.Str::studly($exploded[1]).'::query');
                $current = $query->where('langcode', $langcode)->where('uuid_host', $current->uuid_host)->first();

                if ($current) {
                    throw new NeedRedirectToLayout('edit/'.$current->content_type.'/'.$current->uuid, $langcode);
                }
            }
        }
    }

    private function validateSlug(Slug $slug, string $langcode): void
    {
        $slug = $this->getCorrectSlug($slug, $langcode);
        if ($slug) {
            throw new NeedRedirectToLayout($slug->slug, $langcode);
        }
    }

    /**
     * @param Slug $slug
     * @param string $path
     * @return LayoutPath|null
     */
    protected function checkCondition(Slug $slug, string $path = ''): ?LayoutPath
    {
        $conditionalLayoutPaths = LayoutPath::where('path', 'LIKE', $slug->type.'.%')->get();
        if (count($conditionalLayoutPaths) == 0) {
            return null;
        }

        $exploded = explode('/', $path);
        $model = Str::studly(str_replace('inside_content_', '', $slug->type ?? ''));
        $content = call_user_func('\Inside\Content\Models\Contents\\'.$model.'::find', $slug->uuid);

        if (! $content) {
            return null;
        }

        foreach ($conditionalLayoutPaths as $conditionalLayoutPath) {
            $conditions = explode('.', $conditionalLayoutPath->path);
            if (! isset($conditions[1])) {
                return null;
            }
            $condition = explode(':', $conditions[1]);
            $value = $condition[1] ?? null;

            if (
                $this->isSubpage($condition, $exploded)
                || $this->checkIfContentHasPath($content, $condition, $value)
            ) {
                return $conditionalLayoutPath;
            }
        }

        return null;
    }

    /**
     * @param array $condition
     * @param array $exploded
     * @return bool
     */
    private function isSubpage(array $condition, array $exploded): bool
    {
        if (
            'subpage' === $condition[0]
            && isset($exploded[1])
            && $condition[1] == $exploded[1]
        ) {
            return true;
        }

        return false;
    }

    private function checkIfContentHasPath(mixed $content, array $condition, ?string $value): bool
    {
        if (
            isset($content->{$condition[0]})
            && ! is_iterable($content->{$condition[0]})
            && $content->{$condition[0]} == $value
        ) {
            return true;
        }

        if (
            isset($content->{Str::camel($condition[0])})
            && $content->{Str::camel($condition[0])} instanceof \Illuminate\Database\Eloquent\Collection
            && in_array($value, $content->{Str::camel($condition[0])}->pluck('uuid')->toArray())
        ) {
            return true;
        }

        return false;
    }

    /**
     * @param string $path
     * @return null|mixed
     */
    public function currentUuid(string $path)
    {
        $exploded = explode('/', $path);
        $slug = Slug::where('slug', '=', $exploded[0])->first();
        if ($slug) {
            return $slug->toArray(0);
        }

        return null;
    }

    /**
     * @param array $data
     * @return Layout
     */
    public function create(array $data): Layout
    {
        (new LayoutValidator())->validate($data);
        $layout = new Layout();
        $layout->fill($data)->save();
        foreach ((array) $data['path'] as $path) {
            $layoutPath = new LayoutPath();
            $layoutPath->fill($path)->layout()->associate($layout)->save();
        }

        return $layout;
    }

    /**
     * @param array $data
     * @param int $id
     * @param string $attribute
     * @return Layout
     */
    public function update(array $data, int $id, string $attribute = 'id'): Layout
    {
        (new LayoutValidator())->validate($data, $id);
        $layout = Layout::where($attribute, '=', $id)->firstOrFail();
        $layout->fill($data)->save();

        if (! isset($data['path'])) {
            return $layout;
        }

        LayoutPath::where('layout_id', '=', $id)->delete();

        foreach ((array) $data['path'] as $path) {
            $layoutPath = new LayoutPath();
            $layoutPath->fill($path)->layout()->associate($layout)->save();
        }

        return $layout;
    }

    /**
     * @param int $id
     * @param string $attribute
     * @return mixed
     */
    public function delete(int $id, string $attribute = 'id')
    {
        return LayoutPath::where('layout_id', '=', $id)->delete() && Layout::where($attribute, '=', $id)->delete();
    }
}
