<?php

namespace Inside\Layout\Http\Controllers;

use Illuminate\Auth\AuthenticationException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;
use Inside\Layout\Exceptions\NeedRedirectToLayout;
use Inside\Layout\Http\Controllers\Controller as BaseController;
use Inside\Layout\Models\Layout;
use Inside\Layout\Repositories\Contracts\LayoutRepositoryInterface;
use Inside\Layout\Transformers\LayoutTransformer;
use Inside\Permission\Facades\Permission;
use Inside\Permission\Models\User;

/**
 * @OA\Tag(name="Layout",description="
# Introduction
 *
Il s'agit du service de mémorisation et récupération des mises en page ( Page builder ).
")
 */
class LayoutController extends BaseController
{
    /**
     * @var LayoutRepositoryInterface
     */
    protected $repository;

    /**
     * @param LayoutRepositoryInterface $repository
     */
    public function __construct(LayoutRepositoryInterface $repository)
    {
        $this->repository = $repository;
    }

    /**
     * @return array
     */
    public function build(): array
    {
        if (Cache::has('inside.layout.need_build')) {
            return ['data' => (string) Cache::get('inside.layout.need_build')];
        }

        return ['data' => false];
    }

    /**
     * @param Request $request
     * @return array
     */
    public function current(Request $request): array
    {
        $path = $request->input('path') ?: '';
        $langCode = $request->input('langcode') ?: '';

        try {
            $layout = $this->repository->current($path, $langCode);
        } catch (NeedRedirectToLayout $e) {
            return ['redirect' => $e->getPath(), 'langcode' => $e->getLangcode()];
        }

        $slug = $this->repository->currentUuid($path);

        if ($layout) {
            $data = $this->item($layout, new LayoutTransformer());
            $data['data']['uuid'] = $slug['uuid'] ?? null;
            $data['data']['type'] = $slug['type'] ?? null;

            return $data;
        }

        $publicPaths = Config::get('layout_pathes', []);
        if (! in_array($path, $publicPaths)) {
            abort(404, 'Not found');
        }

        return ['data' => []];
    }

    /**
     * @OA\Schema(
     *     schema="LayoutResult",
     *     type="object",
     *     @OA\Property(
     *          property="data",
     *          description="La liste des layouts",
     *          type="array",
     *          items=@OA\Items(type="object",
     *             @OA\Property(
     *                property="id",
     *               description="Id du layout",
     *               type="integer"
     *             ),
     *             @OA\Property(
     *                property="blocks",
     *               description="Description des blocs",
     *               type="string",
     *               format="json"
     *             ),
     *             @OA\Property(
     *                property="layout",
     *               description="Description du layout de la page",
     *               type="string",
     *               format="json"
     *             ),
     *             @OA\Property(
     *                property="path",
     *               description="Chemin pour accéder à cette page",
     *               type="array",
     *               items=@OA\Items(type="object",
     *                   @OA\Property(
     *                property="path",
     *               description="chemin d'accès",
     *               type="string",
     *               example="home"
     *             ),
     *              @OA\Property(
     *                property="langcode",
     *               description="language associé à ce chemin",
     *               type="string(2)",
     *               example="en"
     *             )
     *               )
     *             ),
     *             @OA\Property(
     *                property="options",
     *               description="Options lié au layout (majoritairement quels sont les types autorisé à l'ajout)",
     *               type="string",
     *               format="json"
     *             )
     *          )
     *     )
     *  )
     *
     * @OA\Get(
     *      path="/layouts",
     *      operationId="layouts",
     *      tags={"Layout"},
     *      summary="Permet de récupérer la liste des layouts actuellement mémoriser en back. [Réservé aux super
     *      administrateurs]",
     *      @OA\Response(
     *          response=200,
     *          description="Les résultats de la liste des layouts au format JSON",
     *          @OA\JsonContent(ref="#/components/schemas/LayoutResult")
     *       ),
     *      @OA\Response(response=400, description="Bad request")
     *     )
     *
     * @return array
     */
    public function list(): array
    {
        $this->checkRight();
        $layouts = Layout::with('paths')->get();

        return $this->collection($layouts, new LayoutTransformer());
    }

    /**
     * @param Request $request
     * @return array
     */
    public function create(Request $request): array
    {
        $this->checkRight();
        $layout = $this->repository->create($request->all());
        Cache::forever('inside.layout.need_build', time());

        return $this->item($layout, new LayoutTransformer());
    }

    /**
     * @param int $id
     * @param Request $request
     * @return array
     */
    public function update(int $id, Request $request): array
    {
        $this->checkRight();
        $layout = $this->repository->update($request->all(), $id);
        Cache::forever('inside.layout.need_build', time());

        return $this->item($layout, new LayoutTransformer());
    }

    /**
     * @param int $id
     * @return array
     */
    public function delete(int $id): array
    {
        $this->checkRight();
        $this->repository->delete($id);
        Cache::forever('inside.layout.need_build', time());

        return ['data' => []];
    }

    /**
     * @return void
     */
    private function checkRight(): void
    {
        if (! Auth::check() || ! Permission::userCanAccessBackofficeEntry('system')) {
            throw new AuthenticationException();
        }
    }
}
