<?php

namespace Inside\Kernel\Authentication;

use Inside\Facades\Package;

final class SingleSingOnRedirectionService
{
    private bool $enabled = false;

    private string $defaultProvider = SingleSignOnProviderEnum::CREDENTIALS;

    private array $availableProviders = [
        SingleSignOnProviderEnum::CREDENTIALS,
    ];

    private array $excludedPaths = [
        '/login',
        '/reconnection',
        '/external-login',
        '/logout',
        '/tv/*?token=*',
        '/page/*',
    ];

    /**
     * TODO: SSO must be registered themselves through the package
     */
    public function __construct()
    {
        Package::has('inside-authentication-okta') && $this->availableProviders[] = SingleSignOnProviderEnum::OKTA;
        Package::has('inside-authentication-saml') && $this->availableProviders[] = SingleSignOnProviderEnum::SAML;
        Package::has('inside-authentication-oauth2') && $this->availableProviders[] = SingleSignOnProviderEnum::OAUTH2;
        Package::has('inside-authentication-google') && $this->availableProviders[] = SingleSignOnProviderEnum::GOOGLE;
        Package::has('inside-authentication-windows') && $this->availableProviders[] = SingleSignOnProviderEnum::WINDOWS;
    }

    public function isEnabled(): bool
    {
        return $this->enabled;
    }

    public function isProtected(string $path): bool
    {
        $path = str($path)->start('/')->toString();

        return collect($this->excludedPaths)->filter(fn (string $excludedPath) => str($path)->is($excludedPath))->isEmpty();
    }

    public function enable(): void
    {
        $this->enabled = true;
    }

    public function disable(): void
    {
        $this->enabled = false;
    }

    public function setProvider(string $provider): self
    {
        if (in_array($provider, $this->availableProviders)) {
            $this->defaultProvider = $provider;
        }

        return $this;
    }

    public function getLoginUrl(): string
    {
        return SingleSignOnProviderEnum::getLoginUrl($this->defaultProvider);
    }

    public function getLogoutUrl(): string
    {
        return SingleSignOnProviderEnum::getLogoutUrl($this->defaultProvider);
    }
}
