<?php

declare(strict_types=1);

namespace Inside\I18n\Services\LiveTranslators;

use Exception;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\ClientException;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Inside\Database\Eloquent\Builder;
use Inside\I18n\Contracts\LiveTranslator;
use Inside\I18n\Models\LiveTranslation;

class DeeplLiveTranslator implements LiveTranslator
{
    private string $uri;

    private string $authKey;

    private ?Client $client = null;

    public function __construct()
    {
        $uri = config('translation.deepl.uri');
        $authKey = config('translation.deepl.auth_key');

        if (! is_string($uri) || empty($uri)) {
            throw new Exception('Invalid configuration: translators.deepl.uri must be a non-empty string');
        }

        if (! is_string($authKey) || empty($authKey)) {
            throw new Exception('Invalid configuration: translators.deepl.auth_key must be a non-empty string');
        }

        $this->uri = $uri;
        $this->authKey = $authKey;
    }

    private function getClient(): Client
    {
        return $this->client ??= new Client(['headers' => ['Authorization' => "DeepL-Auth-Key $this->authKey"]]);
    }

    public function translate(array $texts, string $targetLang, ?string $sourceLang = null): array
    {
        return collect($texts)
          ->chunk(50)
          ->reduce(
              function (array $result, Collection $texts) use ($targetLang, $sourceLang) {
                  try {
                      $response = $this->getClient()->post($this->uri, [
                          'json' => [
                              'text' => $texts->map(fn ($val) => $val ?: '')->values()->all(),
                              'target_lang' => Str::upper($targetLang),
                              'source_lang' => $sourceLang ? Str::upper($sourceLang) : null,
                              'tag_handling' => 'html',
                          ],
                      ]);
                  } catch (ClientException $exception) {
                      $content = json_decode($exception->getResponse()?->getBody()->getContents() ?? '', true);
                      $message = $content['message'] ?? $exception->getMessage();

                      throw new Exception("Deepl translation Failed: $message");
                  }

                  $content = json_decode($response->getBody()->getContents(), true);

                  $translations = collect($content['translations']);

                  return [
                      'text' => collect($result['text'] ?? [])->merge($translations->pluck('text')),
                      'source_lang' => Str::lower($translations->pluck('detected_source_language')->first()),
                      'target_lang' => Str::lower($targetLang),
                      'translator' => 'deepl',
                  ];
              },
              []
          );
    }

    public function getLiveTranslation(array $texts, string $targetLang, ?string $sourceLang = null): LiveTranslation
    {
        $hash = hash_hmac('sha256', (string) json_encode($texts), config('app.key'));

        $translation = LiveTranslation::query()
            ->where('hash', $hash)
            ->where('target_lang', $targetLang)
            ->when($sourceLang, fn (Builder $query, string $sourceLang) => $query->where('source_lang', $sourceLang))
            ->first();

        if ($translation instanceof LiveTranslation) {
            return $translation;
        }

        $translation = $this->translate($texts, $targetLang, $sourceLang);

        return LiveTranslation::create([
            'hash' => $hash,
            'original' => $texts,
            'source_lang' => $translation['source_lang'],
            'target_lang' => $translation['target_lang'],
            'translator' => $translation['translator'],
            'translation' => $translation['text'],
        ]);
    }
}
