<?php

declare(strict_types=1);

namespace Inside\I18n\Repositories;

use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;

abstract class AbstractRepository
{
    /**
     * @var Model
     */
    protected $model;

    /**
     * @return Model
     */
    public function getModel(): Model
    {
        return $this->model;
    }

    /**
     * @param string $id
     * @param array $related
     * @return Builder|Builder|\Illuminate\Database\Eloquent\Collection|Model|Model|null
     */
    public function find(string $id, array $related = [])
    {
        return $this->getModel()->with($related)->find($id);
    }

    public function findTrashed(string $id, array $related = []): Model
    {
        return $this->getModel()->onlyTrashed()->with($related)->find($id);
    }

    /**
     * @param array $related
     * @param int $perPage
     * @return LengthAwarePaginator|Collection
     */
    public function list(array $related = [], int $perPage = 0)
    {
        $results = $this->getModel()->with($related)->orderBy('created_at', 'desc');

        return $perPage ? $results->paginate($perPage) : $results->get();
    }

    public function count(): int
    {
        return $this->getModel()->count();
    }
}
