<?php

declare(strict_types=1);

namespace Inside\I18n\Providers;

use Exception;
use Illuminate\Support\Facades\Schema;
use Illuminate\Translation\TranslationServiceProvider as LaravelTranslationServiceProvider;
use Inside\I18n\Cache\CacheRepositoryFactory;
use Inside\I18n\Console\Commands\AddLanguageCommand;
use Inside\I18n\Console\Commands\CleanupTranslationsCommand;
use Inside\I18n\Console\Commands\ClearCachedTranslationsCommand;
use Inside\I18n\Console\Commands\ClearLiveTranslationsCommand;
use Inside\I18n\Console\Commands\SynchroniseFrontTranslationsCommand;
use Inside\I18n\Console\Commands\SynchroniseTranslationsCommand;
use Inside\I18n\Console\Commands\TranslationsKeysCommand;
use Inside\I18n\Contracts\LiveTranslator;
use Inside\I18n\Loaders\CacheLoader;
use Inside\I18n\Loaders\DatabaseLoader;
use Inside\I18n\Loaders\FileLoader;
use Inside\I18n\Loaders\JsonLoader;
use Inside\I18n\Loaders\Loader;
use Inside\I18n\Models\Language;
use Inside\I18n\Models\Translation;
use Inside\I18n\Repositories\LanguageRepository;
use Inside\I18n\Repositories\TranslationRepository;
use Inside\I18n\Services\FileLoader as InternalFileLoader;
use Inside\I18n\Services\LiveTranslators\DeeplLiveTranslator;
use Inside\I18n\Services\TranslationsService;
use Inside\I18n\Translator\Translator;

/**
 * Class TranslationServiceProvider
 */
final class TranslationServiceProvider extends LaravelTranslationServiceProvider
{
    public function register(): void
    {
        $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/authentication.php', 'authentication');
        $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/translation.php', 'translation');
        $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/scheduler.php', 'scheduler');

        $this->loadMigrationsFrom(__DIR__.'/../../database/migrations');

        parent::register();
        try {
            if (! Schema::hasTable('inside_languages')) {
                return;
            }
        } catch (Exception) {
            return; // Inside not yet installed
        }

        $this->registerCacheRepository();
        $this->registerTranslationLoader();
        $this->registerTranslationTranslator();
        $this->registerCommands();

        $this->app->singleton(
            TranslationsService::class,
            function ($app) {
                $languageRepository = new LanguageRepository(new Language(), $app);

                return new TranslationsService(
                    new TranslationRepository(new Translation(), $app, $languageRepository),
                    $languageRepository
                );
            }
        );

        $this->app->bind(LiveTranslator::class, DeeplLiveTranslator::class);
    }

    public function provides(): array
    {
        return array_merge(parent::provides(), ['inside.translation.cache.repository']);
    }

    protected function registerCacheRepository(): void
    {
        $this->app->singleton(
            'inside.translation.cache.repository',
            function ($app) {
                $cacheStore = $app['cache']->getStore();

                return CacheRepositoryFactory::make($cacheStore, 'inside_translation');
            }
        );
    }

    protected function registerTranslationLoader(): void
    {
        // Override default translation loader
        $this->app->singleton(
            'translation.loader',
            function ($app) {
                $defaultLocale = $app['config']->get('app.locale');
                $laravelLoader = new InternalFileLoader($app['files'], inside_core_path('/resources/lang'));

                $loader = new Loader(
                    $defaultLocale,
                    [
                        new FileLoader($defaultLocale, $laravelLoader),
                        new JsonLoader($defaultLocale, $laravelLoader),
                        new DatabaseLoader($defaultLocale, $app->make(TranslationRepository::class)),
                    ]
                );

                return new CacheLoader($defaultLocale, $this->app['inside.translation.cache.repository'], $loader, 60); // @phpstan-ignore-line
            }
        );
    }

    protected function registerTranslationTranslator(): void
    {
        // Override default translator
        $this->app->singleton(
            'translator',
            function ($app) {
                $loader = $app['translation.loader'];
                $locale = $app['config']['app.locale'];

                $translator = new Translator($loader, $locale);

                $translator->setFallback($app['config']['app.fallback_locale']);

                return $translator;
            }
        );
    }

    protected function registerCommands(): void
    {
        if ($this->app->runningInConsole()) {
            $this->commands(
                [
                    SynchroniseTranslationsCommand::class,
                    SynchroniseFrontTranslationsCommand::class,
                    CleanupTranslationsCommand::class,
                    ClearCachedTranslationsCommand::class,
                    AddLanguageCommand::class,
                    TranslationsKeysCommand::class,
                    ClearLiveTranslationsCommand::class,
                ]
            );
        }
    }

    /**
     * Merge config from
     *
     * @param string $path
     * @param string $key
     */
    protected function mergeRecursiveConfigFrom(string $path, string $key): void
    {
        $config = $this->app['config']->get($key, []); // @phpstan-ignore-line

        $this->app['config']->set($key, array_merge_recursive(require $path, $config)); // @phpstan-ignore-line
    }
}
