<?php

declare(strict_types=1);

namespace Inside\I18n\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Inside\Database\Eloquent\WithEnhancedBuilder;
use Inside\I18n\Models\Traits\Cacheable;

/**
 * Inside\I18n\Models\Translation
 *
 * @property int                               $id
 * @property int                               $language_id
 * @property string                            $namespace
 * @property string                            $group
 * @property string                            $key
 * @property string|null                       $text
 * @property bool                              $draft
 * @property bool                              $locked
 * @property bool                              $exportable
 * @property \Illuminate\Support\Carbon|null   $created_at
 * @property \Illuminate\Support\Carbon|null   $updated_at
 * @property-read mixed                        $code
 * @property-read Language $language
 * @property-read string                       $locale
 *
 * @method static Builder|Translation in($locale)
 * @method static Builder|Translation keyed($key)
 * @method static \Inside\Database\Eloquent\Builder|Translation newModelQuery()
 * @method static \Inside\Database\Eloquent\Builder|Translation newQuery()
 * @method static \Inside\Database\Eloquent\Builder|Translation query()
 * @method static Builder|Translation whereCreatedAt($value)
 * @method static Builder|Translation whereDraft($value)
 * @method static Builder|Translation whereExportable($value)
 * @method static Builder|Translation whereGroup($value)
 * @method static Builder|Translation whereId($value)
 * @method static Builder|Translation whereKey($value)
 * @method static Builder|Translation whereLanguageId($value)
 * @method static Builder|Translation whereLocked($value)
 * @method static Builder|Translation whereNamespace($value)
 * @method static Builder|Translation whereText($value)
 * @method static Builder|Translation whereUpdatedAt($value)
 *
 * @mixin Builder
 */
final class Translation extends Model
{
    use WithEnhancedBuilder;
    use Cacheable;

    protected $table = 'inside_translations';

    protected $fillable = ['language_id', 'namespace', 'group', 'key', 'text', 'draft', 'exportable'];

    protected $casts = [
        'draft'      => 'boolean',
        'locked'     => 'boolean',
        'exportable' => 'boolean',
    ];

    /**
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function language(): \Illuminate\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(Language::class);
    }

    /**
     * @return string
     */
    public function getLocaleAttribute()
    {
        return $this->language->locale;
    }

    /**
     * @return string
     */
    public function getCodeAttribute(): string
    {
        return $this->namespace === '*' ? "{$this->group}.{$this->key}"
            : "{$this->namespace}::{$this->group}.{$this->key}";
    }

    public function validate(): bool
    {
        $this->draft = false;

        return $this->save();
    }

    public function invalidate(): bool
    {
        $this->draft = true;

        return $this->save();
    }

    public function isValid(): bool
    {
        return ! $this->draft;
    }

    public function lockTranslation(): bool
    {
        $this->locked = true;

        return $this->save();
    }

    public function unlockTranslation(): bool
    {
        $this->locked = false;

        return $this->save();
    }

    public function isLocked(): bool
    {
        return $this->locked;
    }

    public function markAsExportable(): bool
    {
        $this->exportable = true;

        return $this->save();
    }

    public function unmarkAsExportable(): bool
    {
        $this->exportable = false;

        return $this->save();
    }

    public function isExportable(): bool
    {
        return $this->exportable;
    }

    /**
     * scope locale
     *
     * @param Builder $query
     * @param string $locale
     * @return Builder
     */
    public function scopeIn(Builder $query, string $locale)
    {
        return $query->whereHas(
            'language',
            function ($query) use ($locale) {
                $query->where('locale', $locale);
            }
        );
    }

    /**
     * search by full key
     *
     * @param Builder $query
     * @param string $key
     * @return Builder
     */
    public function scopeKeyed(Builder $query, string $key): Builder
    {
        $namespace = '*';
        $group = null;
        if (strpos($key, '::') !== false) {
            [$namespace, $key] = explode('::', $key, 2);
        }
        if (strpos($key, '.') !== false) {
            [$group, $key] = explode('.', $key, 2);
        }

        return $query->where('group', $group)->where('namespace', $namespace)->where('key', $key);
    }
}
