<?php

declare(strict_types=1);

namespace Inside\I18n\Models\Traits;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Str;
use Inside\I18n\Models\Translation;

/**
 * Trait Translatable
 *
 * Allow a model to be translatable
 *
 * @property-read null|Model         $translation
 * @property-read Collection|Model[] $translations
 * @property-read string             $translationModel
 * @property-read bool               $useTranslationFallback
 */
trait Translatable
{
    /**
     * Delete translation in cascade when deleting a translatable content
     */
    protected static bool $deleteTranslationsCascade = true;

    /**
     * Boot translatable so save/delete correctly save/delete translation
     */
    public static function bootTranslatable()
    {
        static::saved(
            function (Model $model) {
                /* @var Translatable $model */
                return $model->saveTranslations();
            }
        );

        static::deleting(
            function (Model $model) {
                if (self::$deleteTranslationsCascade === true) {
                    /* @var Translatable $model */
                    $model->deleteTranslations();
                }
            }
        );
    }

    /**
     * Save translations
     *
     * @return bool
     */
    protected function saveTranslations(): bool
    {
        if (! $this->relationLoaded('translations')) {
            return true;
        }

        $saved = true;
        foreach ($this->translations as $translation) {
            if ($saved && $this->isTranslationDirty($translation)) {
                $translation->setAttribute($this->getForeignKey(), $this->getKey());
                $saved = $translation->save();
            }
        }

        return $saved;
    }

    /**
     * Delete translations, optionally only of locale $locales
     *
     * @param null $locales
     * @throws \Exception
     */
    public function deleteTranslations($locales = null)
    {
        if ($locales === null) {
            $translations = $this->translations()->get();
        } else {
            $locales = (array) $locales;
            $translations = $this->translations()->whereIn('locale', $locales)->get();
        }

        $translations->each(
            function (Translation $translation) {
                $translation->delete();
            }
        );
        $this->load('translations');
    }

    /**
     * Override fill to manage translations
     *
     * @param array $attributes
     * @return Translatable
     */
    public function fill(array $attributes)
    {
        foreach ($attributes as $key => $values) {
            if (in_array($key, list_languages())
                && is_array($values)
            ) {
                $this->getTranslationOrNew($key)->fill($values);
                unset($attributes[$key]);
            } else {
                [$attribute, $locale] = $this->getAttributeAndLocale($key);

                if (in_array($locale, list_languages())
                    && $this->isTranslationAttribute($attribute)
                ) {
                    $this->getTranslationOrNew($locale)->fill([$attribute => $values]);
                    unset($attributes[$key]);
                }
            }
        }

        return parent::fill($attributes);
    }

    /**
     * Get an attribute in current locale or if using magic attribute in asked locale
     *
     * @param $key
     * @return mixed|null
     * @example echo $model->{'attribute:es'};
     */
    public function getAttribute($key)
    {
        [$attribute, $locale] = $this->getAttributeAndLocale($key);

        if ($this->isTranslationAttribute($attribute)) {
            if ($this->getTranslation($locale) === null) {
                return $this->getAttributeValue($attribute);
            }

            // Using mutator when needed
            if ($this->hasGetMutator($attribute)) {
                $this->attributes[$attribute] = $this->getAttributeOrFallback($locale, $attribute);

                return $this->getAttributeValue($attribute);
            }

            return $this->getAttributeOrFallback($locale, $attribute);
        }

        return parent::getAttribute($key);
    }

    /**
     * Override attributesToArray to get item in correct locale
     *
     * @return array
     */
    public function attributesToArray()
    {
        $attributes = parent::attributesToArray();

        $hiddenAttributes = $this->getHidden();

        foreach ($this->translatedAttributes as $field) {
            if (in_array($field, $hiddenAttributes)) {
                continue;
            }

            $attributes[$field] = $this->getAttributeOrFallback(null, $field);
        }

        return $attributes;
    }

    /**
     * Get a translation for $locale or default one
     * if it does not exists get a new translation
     *
     * @param string|null $locale
     * @return Model
     */
    public function getTranslationOrNew(?string $locale = null): Model
    {
        $locale = $locale ?: $this->locale();

        if (($translation = $this->getTranslation($locale, false)) === null) {
            $translation = $this->getNewTranslation($locale);
        }

        return $translation;
    }

    /**
     * Get translation for $locale or throw exception if it does not exists !
     *
     * @param string $locale
     * @return Model
     */
    public function getTranslationOrFail(string $locale): Model
    {
        if (($translation = $this->getTranslation($locale, false)) === null) {
            throw (new ModelNotFoundException())->setModel($this->getTranslationModelName(), $locale);
        }

        return $translation;
    }

    /**
     * Get a new translation for $locale
     *
     * @param string $locale
     * @return Model
     */
    public function getNewTranslation(string $locale): Model
    {
        $modelName = $this->getTranslationModelName();

        /** @var Model $translation */
        $translation = new $modelName();
        $translation->setAttribute('locale', $locale);
        $this->translations->add($translation);

        return $translation;
    }

    /**
     * Correctly set attribute for translatable attributes
     *
     * @param $key
     * @param $value
     * @return $this
     */
    public function setAttribute($key, $value)
    {
        [$attribute, $locale] = $this->getAttributeAndLocale($key);

        if ($this->isTranslationAttribute($attribute)) {
            $this->getTranslationOrNew($locale)->{$attribute} = $value;

            return $this;
        }

        return parent::setAttribute($key, $value);
    }

    /**
     * is translation dirty and need to be saved ?
     *
     * @param Model $translation
     * @return bool
     */
    protected function isTranslationDirty(Model $translation): bool
    {
        $dirtyAttributes = $translation->getDirty();
        unset($dirtyAttributes['locale']);

        return ! empty($dirtyAttributes);
    }

    /**
     * Get translation Model Name set on translationModel model attribute
     * If none exists using Default ones
     *
     * @return string
     */
    public function getTranslationModelName(): string
    {
        return $this->translationModel ?: $this->getTranslationModelNameDefault();
    }

    /**
     * Set Default Translation Model Name ( using this model class name with Translation suffix )
     *
     *
     * @return string
     */
    private function getTranslationModelNameDefault(): string
    {
        return get_class($this).'Translation';
    }

    /**
     * Get locale if translation exists or default locale
     *
     * @return string|null
     */
    private function localeOrFallback()
    {
        return $this->useFallback() && ! $this->translations()->where('locale', $this->locale())->exists()
            ? $this->getDefaultLocale() : $this->locale();
    }

    /**
     * Can we use fall back ?
     * default is true, it can be override adding useTranslationFallback property to
     * translatable model
     *
     * @return bool
     */
    private function useFallback(): bool
    {
        if (isset($this->useTranslationFallback) && is_bool($this->useTranslationFallback)) {
            return $this->useTranslationFallback;
        }

        return config('translation.use_fallback', true);
    }

    /**
     * Get translation in locale or fallback if usefallback is on
     *
     * @return HasOne
     */
    public function translation(): HasOne
    {
        return $this->hasOne($this->getTranslationModelName())->where('locale', $this->localeOrFallback());
    }

    /**
     * Get all translations for this item
     *
     * @return HasMany
     */
    public function translations(): HasMany
    {
        return $this->hasMany($this->getTranslationModelName());
    }

    /**
     * Does translation exists in $locale for this item ?
     *
     * @param string|null $locale
     * @return bool
     */
    public function hasTranslation(?string $locale = null): bool
    {
        $locale = $locale ?: $this->locale();

        foreach ($this->translations as $translation) {
            if ($translation->getAttribute('locale') == $locale) {
                return true;
            }
        }

        return false;
    }

    /**
     * Get translation in $locale or default $locale
     *
     * @param string|null $locale
     * @param bool|null   $withFallback
     * @return Model|null
     */
    public function getTranslation(?string $locale = null, bool $withFallback = null): ?Model
    {
        $configFallbackLocale = $this->getDefaultLocale();
        $locale = $locale ?: $this->locale();
        $withFallback = $withFallback === null ? $this->useFallback() : $withFallback;

        if ($translation = $this->getTranslationByLocaleKey($locale)) {
            return $translation;
        }

        if ($withFallback && $configFallbackLocale) {
            if ($translation = $this->getTranslationByLocaleKey($configFallbackLocale)) {
                return $translation;
            }

            if (is_string($configFallbackLocale)
                && $translation = $this->getTranslationByLocaleKey($configFallbackLocale)
            ) {
                return $translation;
            }
        }

        if ($withFallback && $configFallbackLocale === null) {
            $configuredLocales = list_languages();

            foreach ($configuredLocales as $configuredLocale) {
                if ($locale !== $configuredLocale
                    && $configFallbackLocale !== $configuredLocale
                    && $translation = $this->getTranslationByLocaleKey($configuredLocale)
                ) {
                    return $translation;
                }
            }
        }

        return null;
    }

    /**
     * Get all translations for this item as an array
     *
     * @return array
     */
    public function getTranslationsArray(): array
    {
        $translations = [];

        foreach ($this->translations as $translation) {
            foreach ($this->translatedAttributes as $attr) {
                $translations[$translation->locale][$attr] = $translation->{$attr};
            }
        }

        return $translations;
    }

    /**
     * Helper to get translation by $key
     *
     * @param string $key
     * @return Model|null
     */
    private function getTranslationByLocaleKey(string $key): ?Model
    {
        if ($this->relationLoaded('translation')
            && $this->translation
            && $this->translation->getAttribute('locale') == $key
        ) {
            return $this->translation;
        }

        return $this->translations->firstWhere('locale', $key);
    }

    /**
     * Replicate item with its translation
     *
     * @param array|null $except
     * @return Model
     */
    public function replicateWithTranslations(array $except = null): Model
    {
        $newInstance = $this->replicate($except);

        unset($newInstance->translations);
        foreach ($this->translations as $translation) {
            $newTranslation = $translation->replicate();
            $newInstance->translations->add($newTranslation);
        }

        return $newInstance;
    }

    /**
     * Get current locale
     *
     * @return string
     */
    protected function locale(): string
    {
        return App::getLocale();
    }

    /**
     * Get default locale used as a fallback
     *
     * @return string|null
     */
    public function getDefaultLocale(): ?string
    {
        return config('app.locale');
    }

    /**
     * Does key is a translatable attribute ?
     *
     * @param string $key
     * @return bool
     */
    public function isTranslationAttribute(string $key): bool
    {
        return in_array($key, $this->translatedAttributes);
    }

    /**
     * Is $key attribute set ?
     *
     * @param $key
     * @return bool
     */
    public function __isset($key)
    {
        return $this->isTranslationAttribute($key) || parent::__isset($key);
    }

    /// SCOPES

    /**
     * Chargement des traductions
     *
     * @example  Menu::withTranslation()->get();
     *
     * @param Builder $query
     */
    public function scopeWithTranslation(Builder $query)
    {
        $query->with(
            [
                'translations' => function (Relation $query) {
                    if ($this->useFallback()) {
                        $locale = $this->locale();
                        $locales =
                            array_unique([$locale, $this->getDefaultLocale()]);

                        return $query->whereIn($this->getTranslationsTable().'.locale', $locales);
                    }

                    return $query->where($this->getTranslationsTable().'.locale', $this->locale());
                },
            ]
        );
    }

    /**
     * Chargement des trucs au moins traduit dans une langue
     *
     * @example Menu::translated()->get();
     *
     * @param Builder $query
     * @return Builder
     */
    public function scopeTranslated(Builder $query)
    {
        return $query->has('translations');
    }

    /**
     * Chargement des items traduits dans la lange locale ou langue courrante
     *
     * @example Menu::translatedIn('en')->get();
     *
     * @param Builder $query
     * @param string|null                           $locale
     * @return Builder
     */
    public function scopeTranslatedIn(Builder $query, ?string $locale = null)
    {
        $locale = $locale ?: $this->locale();

        return $query->whereHas('translations', function (Builder $q) use ($locale) {
            $q->where($this->getLocaleKey(), '=', $locale);
        });
    }

    /**
     * @example Menu::notTranslatedIn('en')->get();
     *
     * @param Builder $query
     * @param string|null                           $locale
     * @return Builder
     */
    public function scopeNotTranslatedIn(Builder $query, ?string $locale = null)
    {
        $locale = $locale ?: $this->locale();

        return $query->whereDoesntHave('translations', function (Builder $q) use ($locale) {
            $q->where('locale', '=', $locale);
        });
    }

    /**
     * Get Attribute & locale from $key
     *
     * Useful to update one attribute using
     * $model->{'attribute:en'} = 'in english';
     *
     * @param string $key
     * @return array
     */
    private function getAttributeAndLocale(string $key): array
    {
        if (Str::contains($key, ':')) {
            return explode(':', $key);
        }

        return [$key, $this->locale()];
    }

    /**
     * Get attribute in current locale or get fallback ones if not available
     *
     * @param string|null $locale
     * @param string      $attribute
     * @return mixed|null
     */
    private function getAttributeOrFallback(?string $locale, string $attribute)
    {
        $translation = $this->getTranslation($locale);

        if ((! $translation instanceof Model
                || empty($translation->{$attribute}))
            && $this->useFallback()
        ) {
            $translation = $this->getTranslation($this->getDefaultLocale(), false);
        }

        if ($translation instanceof Model) {
            return $translation->{$attribute};
        }

        return null;
    }

    /**
     * Magic helper to get translation table
     *
     * @return string
     */
    private function getTranslationsTable(): string
    {
        return app()->make($this->getTranslationModelName())->getTable();
    }
}
