<?php

declare(strict_types=1);

namespace Inside\I18n\Jobs;

use Illuminate\Filesystem\Filesystem;
use Illuminate\Support\Facades\Log;
use Inside\I18n\Facades\Translation;
use Inside\I18n\Models\Language;
use Inside\Jobs\Job;
use Inside\Support\Str;
use Symfony\Component\Finder\Finder;

/**
 * Class SynchronizeFrontTranslations
 */
final class SynchronizeFrontTranslations extends Job
{
    protected Filesystem $filesystem;

    /**
     * SynchronizeFrontTranslations constructor.
     */
    public function __construct()
    {
        $this->filesystem = new Filesystem();
    }

    public function handle(): void
    {
        $languages = Language::pluck('locale');
        $frontTranslations = collect();
        $frontDirectories = [
            cms_base_path('inside-factory/factory_modules/@maecia'),
            cms_base_path('front/node_modules/@maecia'),
            cms_base_path('front/inside-customs'),
        ];
        foreach ($frontDirectories as $key => $directory) {
            if (! $this->filesystem->exists($directory)) {
                unset($frontDirectories[$key]);
            }
        }

        if (empty($frontDirectories)) {
            Log::error('[synchronisation front] Aucun dossier front détecté');

            return;
        }
        $finder = Finder::create()->files()->ignoreDotFiles(true)->in($frontDirectories)->exclude('node_modules')->name(
            '*.json'
        );
        foreach ($finder->files() as $file) {
            if (0 == preg_match('#/i18n/#', str_replace(DIRECTORY_SEPARATOR, '/', $file->getRealPath()))) {
                continue;
            }

            $locale = $file->getBasename('.'.$file->getExtension());
            $exploded = array_reverse(explode(DIRECTORY_SEPARATOR, $file->getRealPath()));

            $specialGroup = [];
            $isSpecialCommons =
                preg_match('#inside-library/components/([^/]+)/i18n/#', $file->getRealPath(), $specialGroup);

            if ($isSpecialCommons && count($specialGroup) == 2) {
                $specialGroup = $specialGroup[1];
            }
            /**
             * Examples:
             * node_modules/@maecia/inside-auth/i18n/de.json => (de, auth)
             * node_modules/@maecia/inside-users/i18n/fr/components.json => (fr, users.components)
             */
            if (! $languages->contains($locale)) {
                $locale = $exploded[1];
                $parent = $exploded[3];
            } else {
                $parent = $exploded[2];
            }
            $group = Str::camel(str_replace('inside-', '', $parent));
            if ($group === 'library') {
                $group = 'common';
            } elseif ($isSpecialCommons && in_array($specialGroup, ['filters', 'modals', 'systems', 'views'])) {
                $group = 'commons';
            }

            $decoded = collect(json_decode($file->getContents(), true));
            if ($decoded->isEmpty()) {
                continue;
            }

            if (! $frontTranslations->has($locale)) {
                $frontTranslations[$locale] = collect();
            }

            if ($group == 'customs') {
                // This is the custom override
                foreach ($decoded as $group => $overrides) {
                    if (! $frontTranslations[$locale]->has($group)) {
                        $frontTranslations[$locale][$group] = collect();
                    }
                    $frontTranslations[$locale][$group] =
                        $frontTranslations[$locale][$group]->replaceRecursive($overrides);
                }
                continue;
            }
            if (! $frontTranslations[$locale]->has($group)) {
                $frontTranslations[$locale][$group] = collect();
            }
            $frontTranslations[$locale][$group] = $frontTranslations[$locale][$group]->replaceRecursive($decoded);
        }

        foreach ($frontTranslations as $locale => $translations) {
            $this->filesystem->put(
                cms_base_path('vendor/maecia/inside/i18n/front/lang/'.$locale.'.json'),
                $translations->toJson(JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)
            );
        }
        Translation::rebuildCachedTranslations();
    }
}
