<?php

declare(strict_types=1);

namespace Inside\I18n\Http\Controllers;

use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Storage;
use Inside\I18n\Facades\Translation;
use Inside\I18n\Http\Requests\TranslationRequest;
use Inside\I18n\Jobs\ClearCachedTranslations;
use Inside\I18n\Jobs\SynchronizeFrontTranslations;
use Inside\I18n\Repositories\LanguageRepository;
use Inside\I18n\Repositories\TranslationRepository;
use Inside\Support\Str;
use InvalidArgumentException;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use ZipArchive;

/**
 * Class TranslationController
 */
final class TranslationController
{
    public function __construct(
        protected TranslationRepository $repository,
        protected LanguageRepository $languageRepository
    ) {
    }

    /**
     * Clear cached translations
     */
    public function cacheClear(): JsonResponse
    {
        ClearCachedTranslations::dispatch();

        return response()->json(['status' => true]);
    }

    /**
     * Only get overrides
     */
    public function getOverrides(Request $request): JsonResponse
    {
        $filters = $request->get('filters', []);
        if (is_string($filters)) {
            $filters = json_decode($filters, true) ?? [];
        }

        if ($request->get('query') !== null && ! array_key_exists('query', $filters)) {
            $filters['query'] = $request->get('query');
        }

        return response()->json($this->repository->getOverrideItems($filters));
    }

    /**
     * List all translation
     *
     * @throws Exception
     */
    public function index(Request $request): JsonResponse
    {
        $filters = $request->get('filters', []);
        if (is_string($filters)) {
            $filters = json_decode($filters, true) ?? [];
        }

        if ($request->get('query') !== null && ! array_key_exists('query', $filters)) {
            $filters['query'] = $request->get('query');
        }

        return response()->json($this->repository->indexTranslations($filters));
    }

    /**
     * Get possible filters for translations index
     */
    public function getIndexFilters(): JsonResponse
    {
        return response()->json($this->repository->getFilters());
    }

    /**
     * List all translation groups
     *
     * @throws Exception
     */
    public function groupIndex(Request $request): JsonResponse
    {
        $filters = $request->get('filters', []);
        if (is_string($filters)) {
            $filters = json_decode($filters, true) ?? [];
        }
        if ($request->get('query') !== null && ! array_key_exists('query', $filters)) {
            $filters['query'] = $request->get('query');
        }

        return response()->json($this->repository->getAllGroups($filters));
    }

    /**
     * get Front json main translations
     *
     * @throws Exception
     */
    public function getFrontLang(): JsonResponse
    {
        return response()->json($this->repository->getFastFrontFinalWithFallbacksTranslations());
    }

    /**
     * Get translation for $key
     *
     * @throws Exception
     */
    public function show(string $key): JsonResponse
    {
        return response()->json($this->repository->getTranslations($key));
    }

    /**
     * Save a translation
     */
    public function store(TranslationRequest $request): JsonResponse
    {
        try {
            $this->repository->addTranslation(
                $request->json()->get('locale'),
                $request->json()->get('namespace'),
                $request->json()->get('group'),
                $request->json()->get('key'),
                $request->json()->get('text')
            );
        } catch (Exception $e) {
            return new JsonResponse(
                [
                    'success' => false,
                    'error'   => $e->getMessage(),
                ],
                400
            );
        }

        return response()->json(
            [
                'success' => true,
            ]
        );
    }

    /**
     * Update all translations for $key
     */
    public function updateTranslations(Request $request, string $key): JsonResponse
    {
        try {
            $translations = $this->repository->updateTranslationForKey($key, $request->all());
        } catch (Exception $exception) {
            return new JsonResponse(
                [
                    'success' => false,
                    'error'   => $exception->getMessage(),
                ],
                400
            );
        }

        return response()->json(
            [
                'success'      => true,
                'translations' => $translations->mapWithKeys(
                    function ($translation, $key) {
                        return [$translation->locale => $translation];
                    }
                ),
            ]
        );
    }

    /**
     * Delete override by keys for $locale
     */
    public function deleteOverridedTranslationsByKey(Request $request, string $locale): JsonResponse
    {
        try {
            $translations = $this->repository->deleteOverridedTranslationsByKey($request->input('keys'), $locale);
        } catch (Exception $exception) {
            return new JsonResponse(
                [
                    'success' => false,
                    'error'   => $exception->getMessage(),
                ],
                400
            );
        }

        return response()->json(
            [
                'success'      => true,
                'translations' => $translations,
            ]
        );
    }

    /**
     * Update a translation by its $id
     * @deprecated
     */
    public function update(Request $request, int $id): JsonResponse
    {
        try {
            $this->repository->updateTranslation($id, $request->get('text'));
        } catch (Exception $exception) {
            return new JsonResponse(
                [
                    'success' => false,
                    'error'   => $exception->getMessage(),
                ],
                400
            );
        }

        return response()->json(
            [
                'success' => true,
            ]
        );
    }

    /**
     * Delete a translation
     * @deprecated
     */
    public function destroy(int $id): JsonResponse
    {
        try {
            $this->repository->deleteTranslation($id);
        } catch (Exception $exception) {
            return new JsonResponse(
                [
                    'success' => false,
                    'error'   => $exception->getMessage(),
                ],
                400
            );
        }

        return response()->json(
            [
                'success' => true,
            ]
        );
    }

    /**
     * Lock a translation ( can't be changed or deleted anymore )
     */
    public function lock(int $id): JsonResponse
    {
        try {
            $this->repository->lockTranslation($id);
        } catch (Exception $exception) {
            return new JsonResponse(
                [
                    'success' => false,
                    'error'   => $exception->getMessage(),
                ],
                400
            );
        }

        return response()->json(
            [
                'success' => true,
            ]
        );
    }

    /**
     * Unlock a translation
     */
    public function unlock(int $id): JsonResponse
    {
        try {
            $this->repository->unlockTranslation($id);
        } catch (Exception $exception) {
            return new JsonResponse(
                [
                    'success' => false,
                    'error'   => $exception->getMessage(),
                ],
                400
            );
        }

        return response()->json(
            [
                'success' => true,
            ]
        );
    }

    /**
     * Validate a translation
     */
    public function validate(int $id): JsonResponse
    {
        try {
            $this->repository->validateTranslation($id);
        } catch (Exception $exception) {
            return new JsonResponse(
                [
                    'success' => false,
                    'error'   => $exception->getMessage(),
                ],
                400
            );
        }

        return response()->json(
            [
                'success' => true,
            ]
        );
    }

    /**
     * Invalidate a translation
     */
    public function invalidate(int $id): JsonResponse
    {
        try {
            $this->repository->invalidateTranslation($id);
        } catch (Exception $exception) {
            return new JsonResponse(
                [
                    'success' => false,
                    'error'   => $exception->getMessage(),
                ],
                400
            );
        }

        return response()->json(
            [
                'success' => true,
            ]
        );
    }

    /**
     * Synchronize front to json
     */
    public function synchronize(): JsonResponse
    {
        SynchronizeFrontTranslations::dispatch();

        return response()->json(
            [
                'success' => true,
            ]
        );
    }

    /**
     * Export translations from database source
     */
    public function exportDatabase(): BinaryFileResponse
    {
        $data = $this->repository
            ->getOverrideItems(['state' => 'translated'])
            ->makeHidden(
                [
                    'id',
                    'language_id',
                    'language',
                    'draft',
                    'locked',
                    'exportable',
                    'created_at',
                    'updated_at',
                ]
            )->toJson(
                JSON_PRETTY_PRINT
            );

        // Prepare a zip
        $zipFilenameParts = [
            'i18n',
            config('app.code'),
            inside_version(),
            Carbon::now()->timestamp,
            Str::random(64),
        ];
        $zipFilename = implode('_', $zipFilenameParts).'.zip';
        $zipPath = str_replace('/', DIRECTORY_SEPARATOR, storage_path('app/tmp/'.$zipFilename));
        $zipFile = new ZipArchive();
        $zipFile->open($zipPath, ZipArchive::CREATE);
        $zipFile->addFromString('i18n.json', $data);
        $zipFile->close();

        // Download zip
        $response = response()->download(
            $zipPath,
            $zipFilename,
            ['Content-Type: application/octet-stream', 'Content-Length: '.filesize($zipPath)]
        );

        if (! windows_os()) {
            $response->deleteFileAfterSend(true);
        }

        return $response;
    }

    /**
     * Import an exported translation
     */
    public function importDatabase(): JsonResponse
    {
        // Get archive
        $zipPath = request()->get('i18n');
        if (! $zipPath) {
            throw new InvalidArgumentException('Missing i18n file path');
        }
        if (! Storage::disk('local')->exists($zipPath)) {
            throw new InvalidArgumentException('File path does not exist');
        }

        $zipPath = Storage::disk('local')->path($zipPath);
        $zipFile = new ZipArchive();
        $zipFile->open($zipPath, ZipArchive::CREATE);
        $data = $zipFile->getFromName('i18n.json');
        $zipFile->close();
        if (! $data) {
            throw new InvalidArgumentException('Wrong i18n file');
        }
        $data = json_decode($data);
        if (is_null($data)) {
            throw new InvalidArgumentException('Wrong i18n file');
        }

        return response()->json(
            [
                'success' => true,
                'data'    => Translation::importTranslationsFromData($data),
            ]
        );
    }

    /**
     * Mark translation $key for $locale as exportable
     */
    public function markTranslationAsExportable(string $key, string $locale): JsonResponse
    {
        return response()->json(
            [
                'status' => $this->repository->markAsExporable($key, $locale),
            ]
        );
    }

    /**
     * remove mark on translation $key for $locale as non exportable
     */
    public function unmarkTranslationAsExportable(string $key, string $locale): JsonResponse
    {
        return response()->json(
            [
                'status' => $this->repository->unmarkAsExporable($key, $locale),
            ]
        );
    }

    /**
     * Export Database translation marked as exportable for dev
     */
    public function exportDatabaseForDev(): BinaryFileResponse
    {
        // Prepare a zip
        $zipFilenameParts = [
            'i18n',
            config('app.code'),
            inside_version(),
            Carbon::now()->timestamp,
            Str::random(64),
            'dev',
        ];
        $zipFilename = implode('_', $zipFilenameParts).'.zip';
        $zipPath = str_replace('/', DIRECTORY_SEPARATOR, storage_path('app/tmp/'.$zipFilename));
        $zipFile = new ZipArchive();
        $zipFile->open($zipPath, ZipArchive::CREATE);

        $overrideFiles = Translation::getOverridesByFile();
        $zipFile->addFromString(
            'translations.txt',
            $overrideFiles->keys()->implode("\n")
        );

        foreach ($overrideFiles as $file => $data) {
            if (Str::startsWith($file, 'vendor/maecia/inside/lang')) {
                $array = [];
                foreach ($data as $key => $value) {
                    Arr::set($array, Str::after($key, 'front::'), $value);
                }
                $data = collect($array);
            }
            $zipFile->addFromString(
                $file,
                $data->toJson(
                    JSON_PRETTY_PRINT
                )
            );
        }
        $zipFile->close();

        // Download zip
        $response = response()->download(
            $zipPath,
            $zipFilename,
            ['Content-Type: application/octet-stream', 'Content-Length: '.filesize($zipPath)]
        );

        if (! windows_os()) {
            $response->deleteFileAfterSend(true);
        }

        return $response;
    }
}
