<?php

namespace Inside\Documentation\Services;

use Barryvdh\Reflection\DocBlock;
use Barryvdh\Reflection\DocBlock\Context;
use Barryvdh\Reflection\DocBlock\Serializer as DocBlockSerializer;
use Barryvdh\Reflection\DocBlock\Tag;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Support\Str;
use ReflectionClass;
use ReflectionException;
use ReflectionMethod;

/**
 * Class DocumentationHelper
 *
 * Inspired from barryvdh/laravel-ide-helper
 *
 * @package Inside\Content\Services\Docs
 */
class DocumentationHelper
{
    /**
     * @var array Properties
     */
    protected $properties = [];

    /**
     * @var array Methods
     */
    protected $methods = [];

    /**
     * @var array<string, bool>
     */
    protected $nullableColumns = [];

    /**
     * generate Phpdoc for a dynamic class
     *
     * @param string $className
     * @param string $classPath
     * @throws ReflectionException
     */
    public function generateDoc(string $className, string $classPath): void
    {
        include_once $classPath;

        $reflectionClass = new ReflectionClass($className);
        if (!$reflectionClass->isSubclassOf('Illuminate\Database\Eloquent\Model')) {
            return;
        }
        $model = app()->make($className);
        $this->getPropertiesFromTable($model);
        $this->getPropertiesFromMethods($model);
        $this->getSoftDeleteMethods($model);
        $this->createPhpDocs($className, $classPath);
    }

    /**
     * Get properties
     *
     * @param mixed $model
     * @throws ReflectionException
     */
    protected function getPropertiesFromMethods($model): void
    {
        $methods = get_class_methods($model);
        if ($methods) {
            sort($methods);
            foreach ($methods as $method) {
                if (Str::startsWith($method, 'get')
                    && Str::endsWith($method, 'Attribute')
                    && $method !== 'getAttribute'
                ) {
                    //Magic get<name>Attribute
                    $name = Str::snake(substr($method, 3, -9));
                    if (!empty($name)) {
                        $reflection = new ReflectionMethod($model, $method);
                        $type       = $this->getReturnTypeFromDocBlock($reflection);
                        $this->setProperty($name, $type, true, null);
                    }
                } elseif (Str::startsWith($method, 'set')
                    && Str::endsWith($method, 'Attribute')
                    && $method !== 'setAttribute'
                ) {
                    //Magic set<name>Attribute
                    $name = Str::snake(substr($method, 3, -9));
                    if (!empty($name)) {
                        $this->setProperty($name, null, null, true);
                    }
                } elseif (Str::startsWith($method, 'scope') && $method !== 'scopeQuery') {
                    //Magic set<name>Attribute
                    $name = Str::camel(substr($method, 5));
                    if (!empty($name)) {
                        $reflection = new ReflectionMethod($model, $method);
                        $args       = $this->getParameters($reflection);
                        //Remove the first ($query) argument
                        array_shift($args);
                        $this->setMethod($name, '\Illuminate\Database\Eloquent\Builder|\\' . $reflection->class, $args);
                    }
                } elseif (in_array($method, ['query', 'newQuery', 'newModelQuery'])) {
                    $reflection = new \ReflectionClass($model);
                    $builder    = get_class($model->newModelQuery());
                    $this->setMethod($method, "\\{$builder}|\\" . $reflection->getName());
                } elseif (!method_exists('Illuminate\Database\Eloquent\Model', $method)
                    && !Str::startsWith($method, 'get')
                ) {
                    //Use reflection to inspect the code, based on Illuminate/Support/SerializableClosure.php
                    $reflection = new ReflectionMethod($model, $method);
                    // php 7.x type or fallback to docblock
                    $type = $reflection->getReturnType() !== null ? $reflection->getReturnType()->getName() : (string)$this->getReturnTypeFromDocBlock($reflection);

                    if (($fileName = $reflection->getFileName()) === false) {
                        throw new \Exception("Filename not found for model $model");
                    }

                    $file = new \SplFileObject($fileName);
                    $file->seek($reflection->getStartLine() - 1);
                    $code = '';
                    while ($file->key() < $reflection->getEndLine()) {
                        $code .= $file->current();
                        $file->next();
                    }
                    $code  = trim(preg_replace('/\s\s+/', '', $code));
                    $begin = strpos($code, 'function(');
                    $code  = substr($code, (int)$begin, (strrpos($code, '}') - $begin + 1));
                    foreach (
                        [
                            'hasMany'        => '\Illuminate\Database\Eloquent\Relations\HasMany',
                            'hasManyThrough' => '\Illuminate\Database\Eloquent\Relations\HasManyThrough',
                            'belongsToMany'  => '\Illuminate\Database\Eloquent\Relations\BelongsToMany',
                            'hasOne'         => '\Illuminate\Database\Eloquent\Relations\HasOne',
                            'belongsTo'      => '\Illuminate\Database\Eloquent\Relations\BelongsTo',
                            'morphOne'       => '\Illuminate\Database\Eloquent\Relations\MorphOne',
                            'morphTo'        => '\Illuminate\Database\Eloquent\Relations\MorphTo',
                            'morphMany'      => '\Illuminate\Database\Eloquent\Relations\MorphMany',
                            'morphToMany'    => '\Illuminate\Database\Eloquent\Relations\MorphToMany',
                            'morphedByMany'  => '\Illuminate\Database\Eloquent\Relations\MorphToMany',
                        ] as $relation => $impl
                    ) {
                        $search = '$this->' . $relation . '(';
                        if (stripos($code, $search) || stripos($impl, (string)$type) !== false) {
                            //Resolve the relation's model to a Relation object.
                            $methodReflection = new ReflectionMethod($model, $method);
                            if ($methodReflection->getNumberOfParameters()) {
                                continue;
                            }
                            try {
                                $relationObj = $model->$method();
                                if ($relationObj instanceof Relation) {
                                    $relatedModel = '\\' . get_class($relationObj->getRelated());
                                    $relations    = [
                                        'hasManyThrough',
                                        'belongsToMany',
                                        'hasMany',
                                        'morphMany',
                                        'morphToMany',
                                        'morphedByMany',
                                    ];
                                    if (strpos(get_class($relationObj), 'Many') !== false) {
                                        //Collection or array of models (because Collection is Arrayable)
                                        $this->setProperty(
                                            $method,
                                            $this->getCollectionClass($relatedModel) . '|' . $relatedModel . '[]',
                                            true,
                                            null
                                        );
                                        $this->setProperty(
                                            Str::snake($method) . '_count',
                                            'int|null',
                                            true,
                                            false
                                        );
                                    } elseif ($relation === "morphTo") {
                                        // Model isn't specified because relation is polymorphic
                                        $this->setProperty(
                                            $method,
                                            '\Illuminate\Database\Eloquent\Model|\Eloquent',
                                            true,
                                            null
                                        );
                                    } else {
                                        //Single model is returned
                                        $this->setProperty(
                                            $method,
                                            $relatedModel,
                                            true,
                                            null,
                                            '',
                                            $this->isRelationForeignKeyNullable($relationObj)
                                        );
                                    }
                                }
                            } catch (\Exception $e) {
                                // Sometimes Inside Content get relation that don't work ...
                            }
                        }
                    }
                }
            }
        }
    }

    /**
     * Does this is a relation foreign key nullable ?
     *
     * @param \Illuminate\Database\Eloquent\Relations\Relation $relation
     * @return bool
     * @throws ReflectionException
     */
    private function isRelationForeignKeyNullable(Relation $relation)
    {
        $reflectionObj = new \ReflectionObject($relation);
        if (!$reflectionObj->hasProperty('foreignKey')) {
            return false;
        }
        $fkProp = $reflectionObj->getProperty('foreignKey');
        $fkProp->setAccessible(true);

        return isset($this->nullableColumns[$fkProp->getValue($relation)]);
    }

    /**
     * Set a property doc
     *
     * @param string $name
     * @param string|null   $type
     * @param bool|null   $read
     * @param bool|null   $write
     * @param string $comment
     * @param bool   $nullable
     */
    protected function setProperty(string $name, ?string $type = null, ?bool $read = null, ?bool $write = null, string $comment = '', bool $nullable = false): void
    {
        if (!isset($this->properties[$name])) {
            $this->properties[$name]            = [];
            $this->properties[$name]['type']    = 'mixed';
            $this->properties[$name]['read']    = false;
            $this->properties[$name]['write']   = false;
            $this->properties[$name]['comment'] = (string)$comment;
        }
        if ($type !== null) {
            if ($nullable) {
                $type .= '|null';
            }
            $this->properties[$name]['type'] = $type;
        }
        if ($read !== null) {
            $this->properties[$name]['read'] = $read;
        }
        if ($write !== null) {
            $this->properties[$name]['write'] = $write;
        }
    }

    /**
     * Set a method doc
     *
     * @param string $name
     * @param string $type
     * @param array  $arguments
     */
    protected function setMethod(string $name, string $type = '', array $arguments = []): void
    {
        $methods = array_change_key_case($this->methods, CASE_LOWER);
        if (!isset($methods[strtolower($name)])) {
            $this->methods[$name]              = [];
            $this->methods[$name]['type']      = $type;
            $this->methods[$name]['arguments'] = $arguments;
        }
    }

    /**
     * Create the php doc
     *
     * @param string $class
     * @param string $classPath
     * @throws ReflectionException
     */
    protected function createPhpDocs(string $class, string $classPath): void
    {
        $reflection  = new ReflectionClass($class);
        $namespace   = $reflection->getNamespaceName();
        $classname   = $reflection->getShortName();
        $originalDoc = $reflection->getDocComment();
        $keyword     = $this->getClassKeyword($reflection);
        $phpdoc      = new DocBlock('', new Context($namespace));
        $phpdoc->setText((new DocBlock($reflection, new Context($namespace)))->getText());
        if (!$phpdoc->getText()) {
            $phpdoc->setText($class);
        }
        $properties = [];
        $methods    = [];
        foreach ($phpdoc->getTags() as $tag) {
            $name = $tag->getName();
            if ($name == "property" || $name == "property-read" || $name == "property-write") {
                /** @var Tag\ParamTag $tag */
                $properties[] = $tag->getVariableName();
            } elseif ($name == "method") {
                /** @var Tag\MethodTag $tag */
                $methods[] = $tag->getMethodName();
            }
        }
        foreach ($this->properties as $name => $property) {
            $name = "\$$name";
            if (in_array($name, $properties)) {
                continue;
            }
            if ($property['read'] && $property['write']) {
                $attr = 'property';
            } elseif ($property['write']) {
                $attr = 'property-write';
            } else {
                $attr = 'property-read';
            }
            $tagLine = trim("@{$attr} {$property['type']} {$name} {$property['comment']}");
            $tag     = Tag::createInstance($tagLine, $phpdoc);
            $phpdoc->appendTag($tag);
        }
        ksort($this->methods);
        foreach ($this->methods as $name => $method) {
            if (in_array($name, $methods)) {
                continue;
            }
            $arguments = implode(', ', $method['arguments']);
            $tag       = Tag::createInstance("@method static {$method['type']} {$name}({$arguments})", $phpdoc);
            $phpdoc->appendTag($tag);
        }
        if (!$phpdoc->getTagsByName('mixin')) {
            $phpdoc->appendTag(Tag::createInstance("@mixin \\Eloquent", $phpdoc));
        }
        $oaProperties = collect();
        foreach ($this->properties as $name => $property) {
            $oaProperties[] = "@OA\\Property(property=\"".$name."\",\ntitle=\"" .$name . "\",\ntype=\"".$property['type']."\",\ndescription=\"\")";
        }

        $phpdoc->appendTag(Tag::createInstance("@OA\\Schema(schema=\"".$class."\",title=\"".class_to_type($class)."\",type=\"object\",description=\"Dynamic model generated by inside content\",".$oaProperties->implode(",\n").")", $phpdoc));
        $serializer = new DocBlockSerializer();
        $serializer->getDocComment($phpdoc);
        $docComment = $serializer->getDocComment($phpdoc);

        $filename = $reflection->getFileName();
        $contents = file_get_contents($classPath);

        if (!$contents) {
            return;
        }

        if ($originalDoc) {
            $contents = str_replace($originalDoc, $docComment, $contents);
        } else {
            $needle  = "class {$classname}";
            $replace = "{$docComment}\nclass {$classname}";
            $pos     = strpos($contents, $needle);
            if ($pos !== false) {
                $contents = substr_replace($contents, $replace, $pos, strlen($needle));
            }
        }
        file_put_contents($classPath, $contents);
    }

    /**
     * Get the parameters and format them correctly
     *
     * @param ReflectionMethod $method
     * @return array
     */
    public function getParameters(ReflectionMethod $method): array
    {
        //Loop through the default values for paremeters, and make the correct output string
        $params            = [];
        $paramsWithDefault = [];
        foreach ($method->getParameters() as $param) {
            $paramClass = $param->getClass();
            $paramStr   = (!is_null($paramClass) ? '\\' . $paramClass->getName() . ' ' : '') . '$' . $param->getName();
            $params[]   = $paramStr;
            if ($param->isOptional() && $param->isDefaultValueAvailable()) {
                $default = $param->getDefaultValue();
                if (is_bool($default)) {
                    $default = $default ? 'true' : 'false';
                } elseif (is_array($default)) {
                    $default = 'array()';
                } elseif (is_null($default)) {
                    $default = 'null';
                } elseif (is_int($default)) {
                    //$default = $default;
                } else {
                    $default = "'" . trim($default) . "'";
                }
                $paramStr .= " = $default";
            }
            $paramsWithDefault[] = $paramStr;
        }

        return $paramsWithDefault;
    }

    /**
     * Determine a model classes' collection type.
     *
     * @see http://laravel.com/docs/eloquent-collections#custom-collections
     * @param string $className
     * @return string
     */
    private function getCollectionClass($className)
    {
        // Return something in the very very unlikely scenario the model doesn't
        // have a newCollection() method.
        if (!method_exists($className, 'newCollection')) {
            return '\Illuminate\Database\Eloquent\Collection';
        }
        /** @var Model $model */
        $model = new $className();

        return '\\' . get_class($model->newCollection());
    }

    /**
     * Get method return type based on it DocBlock comment
     *
     * @param ReflectionMethod $reflection
     *
     * @return null|string
     */
    protected function getReturnTypeFromDocBlock(ReflectionMethod $reflection): ?string
    {
        $type   = null;
        $phpdoc = new DocBlock($reflection);
        if ($phpdoc->hasTag('return')) {
            /** @var Tag\ReturnTag $returnTag */
            $returnTag = $phpdoc->getTagsByName('return')[0];
            $type = $returnTag->getType();
        }

        return $type;
    }

    /**
     * Generates methods provided by the SoftDeletes trait
     *
     * @param Model $model
     */
    protected function getSoftDeleteMethods(Model $model): void
    {
        $traits = class_uses(get_class($model), true);
        if (is_array($traits) && in_array('Illuminate\\Database\\Eloquent\\SoftDeletes', $traits)) {
            $this->setMethod('forceDelete', 'bool|null', []);
            $this->setMethod('restore', 'bool|null', []);
            $this->setMethod('withTrashed', '\Illuminate\Database\Query\Builder|\\' . get_class($model), []);
            $this->setMethod('withoutTrashed', '\Illuminate\Database\Query\Builder|\\' . get_class($model), []);
            $this->setMethod('onlyTrashed', '\Illuminate\Database\Query\Builder|\\' . get_class($model), []);
        }
    }

    /**
     * @param ReflectionClass $reflection
     * @return string
     */
    private function getClassKeyword(ReflectionClass $reflection)
    {
        if ($reflection->isFinal()) {
            $keyword = 'final ';
        } elseif ($reflection->isAbstract()) {
            $keyword = 'abstract ';
        } else {
            $keyword = '';
        }

        return $keyword;
    }

    protected function getPropertiesFromTable(Model  $model): void
    {
        $table            = $model->getConnection()->getTablePrefix() . $model->getTable();
        $schema           = $model->getConnection()->getDoctrineSchemaManager();
        $databasePlatform = $schema->getDatabasePlatform();
        $databasePlatform->registerDoctrineTypeMapping('enum', 'string');
        $database = null;
        if (strpos($table, '.')) {
            [$database, $table] = explode('.', $table);
        }
        $columns = $schema->listTableColumns($table, $database);
        if ($columns) {
            foreach ($columns as $column) {
                $name = $column->getName();
                if (in_array($name, $model->getDates())) {
                    $type = '\Illuminate\Support\Carbon';
                } else {
                    $type = $column->getType()->getName();
                    switch ($type) {
                        case 'string':
                        case 'text':
                        case 'date':
                        case 'time':
                        case 'guid':
                        case 'datetimetz':
                        case 'datetime':
                            $type = 'string';
                            break;
                        case 'integer':
                        case 'bigint':
                        case 'smallint':
                            $type = 'integer';
                            break;
                        case 'boolean':
                            switch (config('database.default')) {
                                case 'sqlite':
                                case 'mysql':
                                    $type = 'integer';
                                    break;
                                default:
                                    $type = 'boolean';
                                    break;
                            }
                            break;
                        case 'decimal':
                        case 'float':
                            $type = 'float';
                            break;
                        default:
                            $type = 'mixed';
                            break;
                    }
                }
                $comment = $column->getComment();
                if (!$column->getNotnull()) {
                    $this->nullableColumns[$name] = true;
                }
                $this->setProperty($name, $type, true, true, $comment ?? '', !$column->getNotnull());
                $this->setMethod(
                    Str::camel("where_" . $name),
                    '\Illuminate\Database\Eloquent\Builder|\\' . get_class($model),
                    ['$value']
                );
            }
        }
    }
}
