<?php

namespace Inside\Documentation\Services;

use Faker\Factory;
use Faker\Generator;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Bus;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Inside\Authentication\Facades\Authentication;
use Inside\Authentication\Models\User;
use Inside\Content\Exceptions\ValidatorException;
use Inside\Content\Facades\Schema;
use Inside\Content\Jobs\DeleteContent;
use Inside\Host\Bridge\BridgeContentType;
use Inside\Support\Str;
use Intervention\Image\Facades\Image;

/**
 * Class Cypress
 *
 * @package Inside\Documentation\Services
 */
class Cypress
{
    /**
     * @var Generator
     */
    protected $faker;

    /**
     * @var Application|null
     */
    protected $app;

    /**
     * Cypress constructor.
     *
     * @param Application|null $app
     */
    public function __construct(?Application $app)
    {
        $this->app   = $app;
        $this->faker = $this->makeFaker();
    }

    /**
     * Create a random content data of $type
     *
     * @param  string  $type
     *
     * @return array
     */
    public function factory(string $type): array
    {
        $data   = [];
        $locale = $this->faker()->randomElement(list_languages());
        // Reload faker in $locale
        $this->faker = $this->makeFaker($locale);
        foreach (Schema::getFieldListing($type) as $fieldName) {
            if ($fieldName == 'comments') {
                continue;
            }
            $options = Schema::getFieldOptions($type, $fieldName);

            // users exceptions
            if ($type === 'users') {
                if ($fieldName == 'email') {
                    $data[$fieldName] = $this->faker()->safeEmail();
                    continue;
                } elseif ($fieldName == 'firstname') {
                    $data[$fieldName] = $this->faker()->firstName();
                    continue;
                } elseif ($fieldName == 'lastname') {
                    $data[$fieldName] = $this->faker()->lastName();
                    continue;
                } elseif ($fieldName == 'password') {
                    $data[$fieldName] = $this->faker()->password(12, 24);
                    continue;
                } elseif ($fieldName == 'name') {
                    $data[$fieldName] = $this->faker()->name();
                    continue;
                }
            }
            if ($fieldName == 'langcode') {
                $data[$fieldName] = $locale;
                continue;
            }
            $data[$fieldName] = $this->getFakeFieldValue($fieldName, $options);
        }

        return $data;
    }

    /**
     * Prepare a random image and get its relative path
     *
     * @param  string  $extension
     *
     * @return string
     * @throws \Exception
     */
    public function getRandomImage(string $extension): string
    {
        if (!in_array($extension, ['jpg', 'png', 'gif', 'jpeg'])) {
            throw new \Exception('Not an image extension');
        }
        if ($extension == 'jpg') {
            $extension = 'jpeg';
        }
        $storage       = Storage::disk('local');
        $fakeDirectory = 'fakes'.DIRECTORY_SEPARATOR.now()->timestamp;
        $tempDirectory = 'fakes'.DIRECTORY_SEPARATOR.'tmp';
        if (!$storage->exists($tempDirectory)) {
            $storage->makeDirectory($tempDirectory);
        }
        if (!$storage->exists($fakeDirectory)) {
            $storage->makeDirectory($fakeDirectory);
        }
        $fileName = Str::slug($this->faker()->word).'.'.$extension;
        $tmpImage = $this->faker()
            ->image($storage->path($tempDirectory), 640, 480, null, false);
        $path     = $fakeDirectory.DIRECTORY_SEPARATOR.$fileName;
        if (File::extension($tmpImage) == $extension) {
            if (!$storage->move(
                $tempDirectory.DIRECTORY_SEPARATOR.$tmpImage,
                $path
            )
            ) {
                Log::debug('Can not move image from ['.$tempDirectory.'] to ['
                    .$path.']');
                throw new \Exception('Can not generate image');
            }
        } else {
            Image::make($storage->path($tempDirectory.DIRECTORY_SEPARATOR
                .$tmpImage))->save($storage->path($path));
        }
        Log::debug('path ['.$path.'] ('.($storage->exists($path) ? 'OUI'
                : 'NON').')');

        return $path;
    }

    /**
     * Get a random file of type $extension and get its relative path
     *
     * @param  string  $extension
     *
     * @return string
     * @throws \Exception
     */
    public function getRandomFile(string $extension): string
    {
        $extension = Str::lower($extension);
        if (in_array($extension, ['jpg', 'jpeg', 'gif', 'png'])) {
            return $this->getRandomImage($extension);
        }

        $directory
            = cms_base_path('vendor/maecia/inside/documentation/resources/assets/fakes/'
            .$extension);
        if (!File::exists($directory) || !File::isDirectory($directory)) {
            throw new \Exception('type ['.$extension.'] does not exists');
        }
        $storage       = Storage::disk('local');
        $fakeDirectory = 'fakes'.DIRECTORY_SEPARATOR.now()->format('YmdHis');
        $tempDirectory = 'fakes'.DIRECTORY_SEPARATOR.'tmp';
        if (!$storage->exists($tempDirectory)) {
            $storage->makeDirectory($tempDirectory);
        }
        if (!$storage->exists($fakeDirectory)) {
            $storage->makeDirectory($fakeDirectory);
        }

        $fileName = Str::slug($this->faker()->word).'.'.$extension;
        $tmpFile  = $this->faker()
            ->file($directory, $storage->path($tempDirectory), false);
        $path     = $fakeDirectory.DIRECTORY_SEPARATOR.$fileName;
        $storage->move($tempDirectory.DIRECTORY_SEPARATOR.$tmpFile, $path);

        return $path;
    }

    /**
     * Update content type config
     *
     * @param  string  $type
     * @param  array  $options
     *
     * @return bool
     */
    public function updateContentTypeConfig(string $type, array $options): bool
    {
        $bridge = new BridgeContentType();

        try {
            $bridge->updateContentTypeOptions($type, $options);
        } catch (\Throwable $e) {
            return false;
        }

        return true;
    }

    /**
     * Update field config
     *
     * @param  string  $type
     * @param  string  $fieldName
     * @param  array  $options
     *
     * @return bool
     */
    public function updateFieldConfig(
        string $type,
        string $fieldName,
        array $options
    ): bool {
        $bridge = new BridgeContentType();

        try {
            $bridge->updateFieldOptions($type, $fieldName, $options);
        } catch (\Throwable $e) {
            return false;
        }

        return true;
    }

    /**
     * cleanup temp user and its content
     *
     * @param  \Inside\Authentication\Models\User  $user
     * @param  bool  $logout
     *
     * @return array
     */
    public function cleanUp(User $user, bool $logout = true): array
    {
        $toDelete = [];
        $userUuid = $user->uuid;
        if ($logout) {
            Authentication::logout($user);
        }
        if ($user->email !== null && Str::endsWith($user->email, '@maecia.com')) {
            throw ValidatorException::withMessages(['warning' => 'Utilisation de cy.cleanup avec un api token d\'un utilisateur maecia']);
        }

        $deleteJobs = [];
        foreach (Schema::getContentTypes() as $contentType) {
            if ($contentType === 'users') {
                continue;
            }
            $query = call_user_func(type_to_class($contentType).'::query');
            $query->where('author', $userUuid);
            $toDelete[$contentType] = [];
            foreach ($query->pluck('uuid') as $uuid) {
                $toDelete[$contentType][] = $uuid;
                $deleteJobs[]  = new DeleteContent($contentType, $uuid);
            }
        }
        $toDelete['users'] = [$userUuid];
        $deleteJobs[]  = new DeleteContent('users', $userUuid);

        Bus::chain($deleteJobs)->dispatch();

        return $toDelete;
    }

    /**
     * Pick random ( $max ) contents of $type
     *
     * @param  string $type
     * @param  int  $max
     *
     * @return array
     */
    protected function pickRandomContents(string $type, int $max = 1): array
    {
        $query = call_user_func(type_to_class($type).'::query');

        return $query->whereStatus(true)->inRandomOrder()->pluck('uuid')
            ->take(Arr::random(range(1, $max)))->all();
    }

    /**
     * Get fake field value using $options
     *
     * @param  string  $fieldName
     * @param  array  $options
     *
     * @return array|bool|mixed|string|null
     * @throws \Exception
     */
    protected function getFakeFieldValue(string $fieldName, array $options)
    {
        switch ($options['type']) {
            case 'list_float':
            case 'list_integer':
            case 'list_string':
                return $this->faker()
                    ->randomElement(array_keys($options['allowed_values'][config(
                        'app.locale',
                        Factory::DEFAULT_LOCALE
                    )]));
            case 'textarea':
                return $this->faker()->paragraphs(3, true);
            case 'wysiwyg':
                return $this->faker()->randomHtml();
            case 'text':
                if (Str::contains($fieldName, 'mobile')) {
                    try {
                        return $this->faker()->mobileNumber;
                    } catch (\InvalidArgumentException $e) {
                        return $this->faker()->phoneNumber;
                    }
                } elseif (Str::contains($fieldName, 'phone')) {
                    return $this->faker()->phoneNumber;
                }
                switch ($options['widget']) {
                    case 'inside_link_field_widget':
                        return $this->faker()->url;
                    case 'inside_phone_field_widget':
                        return $this->faker()->phoneNumber;
                    case 'inside_color_picker_field_widget':
                        return $this->faker()->hexColor;
                    case 'inside_icon_picker_field_widget':
                        return []; // TODO
                }

                return $this->faker()->text($options['size'] ?? 255);
            case 'section':
                return $this->getFakeSections(
                    $options['target'],
                    $this->faker()->numberBetween(0, 6)
                );
            case 'checkbox':
            case 'boolean':
                return $this->faker()->boolean(60);
            case 'timestamp':
                return $this->faker()->dateTimeBetween('-1 year')
                    ->format('Y-m-d H:i:s');
            case 'reference':
                return $this->pickRandomContents(
                    Arr::first($options['target']),
                    $options['cardinality'] === -1 ? 6 : 1
                );
            case 'image':
                return $this->getRandomImage($this->faker()
                    ->randomElement(explode(' ', $options['extensions'])));
            case 'file':
                return $this->getRandomFile($this->faker()
                    ->randomElement(explode(' ', $options['extensions'])));
        }

        return null;
    }

    /**
     * get $count fake sections of types $sectionTypes
     *
     * @param  array  $sectionTypes
     * @param  int  $count
     *
     * @return array|null
     */
    protected function getFakeSections(array $sectionTypes, int $count): ?array
    {
        if ($count == 0) {
            return null;
        }
        $sections = [];

        for ($i = 0; $i < $count; $i++) {
            $type              = $this->faker()->randomElement($sectionTypes);
            $section           = [];
            $section['bundle'] = $type;
            foreach (Schema::getFieldListing($type) as $fieldName) {
                if ($fieldName == 'comments') {
                    continue;
                }
                $options             = Schema::getFieldOptions(
                    $type,
                    $fieldName
                );
                $section[$fieldName] = $this->getFakeFieldValue(
                    $fieldName,
                    $options
                );
            }
            $sections[] = $section;
        }

        return $sections;
    }

    /**
     * Make a faker in $local language
     *
     * @param  string|null  $locale
     *
     * @return Generator
     */
    protected function makeFaker(?string $locale = null): Generator
    {
        $locale = $locale ?? config('app.locale', Factory::DEFAULT_LOCALE);

        if (isset($this->app) && $this->app->bound(Generator::class)) {
            return $this->app->make(Generator::class, ['locale' => $locale]);
        }

        return Factory::create($locale);
    }

    /**
     * Get our faker
     *
     * @param  string|null  $locale
     *
     * @return Generator
     */
    protected function faker(?string $locale = null): Generator
    {
        return is_null($locale) ? $this->faker : $this->makeFaker($locale);
    }
}
