<?php

namespace Inside\Documentation\Http\Controllers;

use Illuminate\Http\Response;
use Illuminate\Support\Facades\File;
use Inside\Documentation\Services\Generator;
use Laravel\Lumen\Routing\Controller;
use OpenApi\Annotations as OA;

/**
 * @OA\Info(
 *      version="2.0.0",
 *      title="Inside 2 API",
 *      description="| Il s'agit de la documentation de l'api d'Inside 2.",
 *      @OA\Contact(
 *          email="technique@maecia.com"
 *      ),
 *     @OA\License(
 *         name="Commerciale"
 *     ),
 *     x={
 *        "logo": {
 *               "url": "https://dev.sid2.pp.intranet-inside.com/front/factory_modules/inside-library/assets/img/logos/logo.png",
 *               "altText": "Inside 2"
 *              }
 *     }
 * )
 * @OA\SecurityScheme(
 *     securityScheme="apiToken",
 *     type="apiKey",
 *     in="header",
 *     name="api-token"
 * )
 * @OA\OpenApi(
 *   security={
 *     {
 *        "apiToken": {}
 *     }
 *  }
 *
 * )
 */

/**
 * @OA\Schema(
 *     schema="BadRequest",
 *     type="object",
 *     @OA\Property(
 *        property="error",
 *        description="L'erreur conduisant au Bad request",
 *        type="string",
 *        example="The given data was invalid."
 *     )
 * )
 * @OA\Schema(
 *     schema="UnprocessableEntity",
 *     type="object",
 *     additionalProperties=@OA\AdditionalProperties(
 *     type="array",
 *     description="Un tableau d'erreur",
 *     items=@OA\Items(type="string")
 *      ),
 *     example={"email":{"L'email est invalide","L'email est déjà utilisée."},"username":{"Votre identifiant est vide."}}
 * )
 */

/**
 *
 * @OA\Server(
 *      url=DEV_DOCUMENTATION_HOST,
 *      description="Serveur de développement"
 * )

 * @OA\Server(
 *      url=DOCUMENTATION_HOST,
 *      description="Serveur de production"
 *  )
 *
 */
class DocumentationController extends Controller
{
    /**
     * Get documentation using redoc
     *
     * @return array|\Illuminate\View\View|mixed
     */
    public function index()
    {
        return view(
            'documentation::redoc',
            [
                'urlToDocs'        => route(
                    'documentation.docs',
                    ['jsonFile' => config('documentation.paths.docs_json', 'api-docs.json')]
                ),
                'operationsSorter' => config('documentation.operations_sort'),
                'configUrl'        => config('documentation.additional_config_url'),
                'validatorUrl'     => config('documentation.validator_url'),
            ]
        );
    }

    /**
     * Get documentation using swagger
     *
     * @return array|\Illuminate\View\View|mixed
     */
    public function swagger()
    {
        return view(
            'documentation::swagger',
            [
                'urlToDocs'        => route(
                    'documentation.docs',
                    ['jsonFile' => config('documentation.paths.docs_json', 'api-docs.json')]
                ),
                'operationsSorter' => config('documentation.operations_sort'),
                'configUrl'        => config('documentation.additional_config_url'),
                'validatorUrl'     => config('documentation.validator_url'),
            ]
        );
    }

    /**
     * Get the doc file in json format
     *
     * @param null $jsonFile
     * @return \Illuminate\Http\Response
     */
    public function docs($jsonFile = null)
    {
        $filePath = config('documentation.paths.docs') . '/' . (!is_null($jsonFile)
                ? $jsonFile
                : config(
                    'documentation.paths.docs_json',
                    'api-docs.json'
                ));
        //  if (!File::exists($filePath)) {
        try {
            Generator::generateDocs();
        } catch (\Exception $e) {
            abort(404, 'Cannot find ' . $filePath . ' and cannot be generated.');
        }
        //}
        $content = File::get($filePath);

        return (new Response(
            $content,
            200,
            [
                        'Content-Type' => 'application/json',
                    ]
        ));
    }
}
