<?php

use Illuminate\Contracts\Auth\Factory as AuthFactory;
use Illuminate\Contracts\Broadcasting\Factory as BroadcastFactory;
use Illuminate\Contracts\Support\Htmlable;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Inside\Authentication\Models\User as AuthUser;
use Inside\Content\Facades\ContentHelper;
use Inside\Support\Contracts\DeferringDisplayableValue;
use Inside\Support\Str;

if (! function_exists('config_path')) {
    /**
     * Get our global config path
     *
     * @param string $path
     *
     * @return string
     */
    function config_path(string $path = ''): string
    {
        /** @var string $basePath */
        $basePath = inside_core_path('config');

        return $basePath.($path ? DIRECTORY_SEPARATOR.$path : $path);
    }
}

if (! function_exists('back_path')) {
    /**
     * Get our back specific module path
     *
     * @param string $path
     *
     * @return string
     */
    function back_path($path = ''): string
    {
        /** @var string $appCode */
        $appCode = str_replace('_dev', '', env('APP_CODE'));

        /** @var string $basePath */
        $basePath = realpath(__DIR__.'/../../../'.$appCode.'-back');

        // Fallback to inside
        if (! $basePath) {
            /** @var string $basePath */
            $basePath = inside_core_path();
        }

        return $basePath.($path ? DIRECTORY_SEPARATOR.$path : $path);
    }
}

if (! function_exists('inside_base_path')) {
    /**
     * Get inside module base path
     *
     * @param string $path
     *
     * @return string
     */
    function inside_base_path(string $path = ''): string
    {
        return app()->basePath().($path ? DIRECTORY_SEPARATOR.$path : $path);
    }
}

if (! function_exists('inside_core_path')) {
    /**
     * Get inside module core path
     *
     * @param string $path
     *
     * @return string
     */
    function inside_core_path(string $path = ''): string
    {
        return app()->corePath().($path ? DIRECTORY_SEPARATOR.$path : $path);
    }
}

if (! function_exists('cms_public_path')) {
    /**
     * Get cms public path
     *
     * @param string $path
     *
     * @return string
     */
    function cms_public_path(string $path = ''): string
    {
        $cmsPublicPath = cms_base_path('public');

        return $cmsPublicPath.($path ? DIRECTORY_SEPARATOR.$path : $path);
    }
}

if (! function_exists('cms_base_path')) {
    /**
     * Get cms base path
     *
     * @param string $path
     *
     * @return string
     */
    function cms_base_path(string $path = ''): string
    {
        /** @var string $cmsBasePath */
        $cmsBasePath = realpath(__DIR__.'/../../../../..');

        return $cmsBasePath.($path ? DIRECTORY_SEPARATOR.$path : $path);
    }
}

if (! function_exists('theme_path')) {
    /**
     * Get theme absolute path
     *
     * @param string $path
     * @return string
     */
    function theme_path(string $path): string
    {
        return cms_base_path('themes/custom/inside-drupal-theme').($path ? DIRECTORY_SEPARATOR.$path : $path);
    }
}

if (! function_exists('set_response')) {
    /**
     * Correcly set data to a response,
     * could be a simple Response, or a JsonResponse, etc...
     */
    function set_response(Symfony\Component\HttpFoundation\Response $response, mixed $data): void
    {
        if ($response instanceof JsonResponse) {
            $response->setData($data);
        } elseif ($response instanceof Response) {
            $response->setContent($data);
        }
    }
}

if (! function_exists('json_decode_response')) {
    /**
     * Correcly get data from a response.
     *
     * @param JsonResponse|RedirectResponse|Response|mixed $response
     *
     * @return JsonResponse|RedirectResponse|Response|mixed|null
     */
    function json_decode_response($response)
    {
        if ($response instanceof JsonResponse) {
            return $response->getData(true);
        } elseif ($response instanceof RedirectResponse) {
            return null; // On our middleware, we return directly response if result is null
        } elseif ($response instanceof Response) {
            return json_decode($response->getContent(), true);
        }

        return $response;
    }
}

if (! function_exists('is_maecia_admin')) {
    function is_maecia_admin(?string $uuid = null): bool
    {
        $user = $uuid === null ? Auth::user() : AuthUser::find($uuid);
        if (! $user instanceof AuthUser || is_null($user->email)) {
            return false;
        }

        return $user->hasRole('super_administrator') && Str::endsWith($user->email, '@maecia.com');
    }
}

if (! function_exists('get_default_queue_name')) {
    function get_default_queue_name(): ?string
    {
        if (env('QUEUE_DRIVER', 'redis') != 'sync') {
            return config(
                'queue.connections.'.config('queue.default', 'sync').'.queue',
                env('APP_CODE', 'sid2').'_default'
            );
        }

        return null;
    }
}

if (! function_exists('get_indexation_queue_name')) {
    function get_indexation_queue_name(): ?string
    {
        if (env('SCOUT_QUEUE', true) !== null) {
            return config('scout.queue.queue', env('APP_CODE', 'sid2').'_indexation');
        }

        return null;
    }
}

if (! function_exists('get_lts_queue_name')) {
    function get_lts_queue_name(): ?string
    {
        return get_default_queue_name();
    }
}

if (! function_exists('get_high_priority_queue_name')) {
    function get_high_priority_queue_name(): ?string
    {
        return get_default_queue_name();
    }
}

if (! function_exists('get_low_priority_queue_name')) {
    function get_low_priority_queue_name(): ?string
    {
        return get_default_queue_name();
    }
}

if (! function_exists('get_images_queue_name')) {
    function get_images_queue_name(): string
    {
        $default = config('app.code').(config('app.debug', false) ? '_dev' : '').'_images';

        return env('IMAGES_QUEUE', $default);
    }
}

if (! function_exists('inside_version')) {
    /**
     * Try to get current inside version
     *
     * @return string
     */
    function inside_version(): string
    {
        $version = 'master';
        if (function_exists('setting')) {
            $SystemVersion = json_decode(setting('system', 'version'), true);
            if ($SystemVersion !== null) {
                $version = (object) $SystemVersion;
                $version = "{$version->major}.{$version->minor}.{$version->patch}";
            }
        }

        return $version;
    }
}
if (! function_exists('inside_bytes_to_human')) {
    /**
     * A simple helper to get bytes human reading string
     *
     * @param int $bytes
     * @param int $precision
     * @return string
     */
    function inside_bytes_to_human(int $bytes, int $precision = 2): string
    {
        if ($bytes < 0) {
            throw new InvalidArgumentException('[inside_bytes_to_human] Wrong bytes ['.$bytes.']');
        }

        /** @var AuthUser $me */
        $me = Auth::user();
        $locale = $me->langcode ?? config('app.locale');
        if ($locale == 'fr') {
            $units = ['o', 'Ko', 'Mo', 'Go', 'To', 'Po'];
        } else {
            $units = ['B', 'KiB', 'MiB', 'GiB', 'TiB', 'PiB'];
        }

        for ($i = 0; $bytes > 1024; $i++) {
            $bytes /= 1024;
        }

        return round($bytes, $precision).' '.$units[$i];
    }
}

if (! function_exists('inside_mix')) {
    /**
     * Get path collection for a mix type
     *
     * @param string $type
     * @param string $manifestDirectory
     * @return mixed
     *
     * @throws Exception
     */
    function inside_mix($type, $manifestDirectory = '')
    {
        return app(Inside\Mix::class)(...func_get_args());
    }
}

if (! function_exists('__')) {
    /**
     * Translate the given message.
     *
     * @param  string|null  $key
     * @param  array  $replace
     * @param  string|null  $locale
     * @return mixed
     */
    function __($key = null, $replace = [], $locale = null)
    {
        if (is_null($key)) {
            return $key;
        }

        return Lang::get($key, $replace, $locale);
    }
}

if (! function_exists('redirect')) {
    /**
     * Get an instance of the redirector.
     *
     * @param  string|null  $to
     * @param  int  $status
     * @param  array  $headers
     * @param  bool|null  $secure
     * @return RedirectResponse
     */
    function redirect($to = '/', $status = 302, $headers = [], $secure = null)
    {
        $redirector = new Laravel\Lumen\Http\Redirector(app());

        return $redirector->to($to, $status, $headers, $secure);
    }
}

if (! function_exists('now')) {
    /**
     * Create a new Carbon instance for the current time.
     *
     * @param DateTimeZone|string|null $tz
     * @return Carbon
     */
    function now($tz = null)
    {
        return Carbon::now($tz);
    }
}

if (! function_exists('today')) {
    /**
     * Create a new Carbon instance for the current date.
     *
     * @param DateTimeZone|string|null $tz
     * @return Carbon
     */
    function today($tz = null)
    {
        return Carbon::today($tz);
    }
}

if (! function_exists('list_languages')) {
    /**
     * Get all languages listed in env file.
     *
     * @return string[]
     */
    function list_languages(): array
    {
        return array_map('trim', explode(',', env('LANGUAGES', 'fr')));
    }
}

if (! function_exists('bcrypt')) {
    /**
     * Hash the given value against the bcrypt algorithm.
     *
     * @param string $value
     * @param array $options
     *
     * @return string
     */
    function bcrypt($value, $options = [])
    {
        return app('hash')->make($value, $options);
    }
}

if (! function_exists('request')) {
    /**
     * Get an instance of the current request or an input item from the request.
     *
     * @param  list<string>|string|null  $key
     * @param  mixed  $default
     * @return ($key is null ? \Illuminate\Http\Request : ($key is string ? mixed : array<string, mixed>))
     */
    function request($key = null, $default = null)
    {
        if (is_null($key)) {
            return app('request');
        }

        if (is_array($key)) {
            return app('request')->only($key);
        }

        $value = app('request')->__get($key);

        return is_null($value) ? value($default) : $value;
    }
}

if (! function_exists('validate')) {
    function validate(
        Request $request,
        array $rules,
        array $messages = [],
        array $customAttributes = []
    ): array {
        Validator::make($request->all(), $rules, $messages, $customAttributes)->validate();

        return $request->only(
            collect($rules)->keys()->map(function ($rule) {
                return Illuminate\Support\Str::contains($rule, '.') ? explode('.', $rule)[0] : $rule;
            })->unique()->toArray()
        );
    }
}

if (! function_exists('asset')) {
    /**
     * get asset url
     *
     * @param string $path path to the asset
     * @param bool $onlyPath get it relative instead of full path
     *
     * @return string
     */
    function asset($path, $onlyPath = true)
    {
        if (Illuminate\Support\Str::startsWith(env('APP_URL'), 'https://')) {
            $path = app('url')->secureAsset(str_replace(DIRECTORY_SEPARATOR, '/', $path));
        } else {
            $path = app('url')->asset(str_replace(DIRECTORY_SEPARATOR, '/', $path));
        }
        if ($onlyPath) {
            if (Illuminate\Support\Str::startsWith($path, env('APP_URL'))) {
                $path = Illuminate\Support\Str::after($path, env('APP_URL'));
            }
        }

        return $path;
    }
}

if (! function_exists('inside_link_is_external')) {
    /**
     * Determine if a link is external
     * NOTE: uri is changed to a full external uri if it was a partial uri (eg: www.apple.fr, www.google.fr )
     *
     * @param string $uri
     * @return bool
     */
    function inside_link_is_external(string &$uri): bool
    {
        /** @var array $info */
        $info = parse_url($uri);

        if (array_key_exists('scheme', $info) && $info['scheme'] === 'mailto') {
            return true;
        }

        // Check incomplete uri
        if (
            (! array_key_exists('scheme', $info) || empty($info['scheme']))
            && (array_key_exists('path', $info) && ! empty($info['path']) && $info['path'] != '/')
            && strpos($info['path'], '.') !== false
        ) {
            // We got an incomplete uri, let's complete

            $uri = 'https://'.$uri;

            return true;
        }

        /** @var string $host */
        $host = parse_url(config('app.url'), PHP_URL_HOST);

        return isset($info['host']) && ! empty($info['host'])
            && strcasecmp(
                $info['host'],
                $host
            ) !== 0;
    }
}

if (! function_exists('broadcast')) {
    /**
     * Begin broadcasting an event.
     *
     * @param mixed|null $event
     * @return Illuminate\Broadcasting\PendingBroadcast
     */
    function broadcast($event = null)
    {
        return app(BroadcastFactory::class)->event($event);
    }
}

if (! function_exists('ensure_directory_exists')) {
    function ensure_directory_exists(string $path, int $mode = 0755, bool $recursive = true): void
    {
        $filesystem = new Illuminate\Filesystem\Filesystem();
        if (! $filesystem->isDirectory($path)) {
            $filesystem->makeDirectory($path, $mode, $recursive);
        }
    }
}

if (! function_exists('auth')) {
    /**
     * @param string|null $guard
     * @return mixed
     */
    function auth(?string $guard = null)
    {
        if (is_null($guard)) {
            return app(AuthFactory::class);
        }

        return app(AuthFactory::class)->guard($guard);
    }
}

if (! function_exists('jtrans_choice')) {
    /**
     * choice on json strings
     *
     * @param string $key
     * @param int $count
     * @return string
     */
    function jtrans_choice(string $key, int $count): string
    {
        /** @var string $key */
        $key = __($key);

        return trans_choice($key, $count);
    }
}

if (! function_exists('icollect')) {
    /**
     * Create a collection from the given value.
     *
     * @param mixed $value
     * @return Collection
     * @deprecated use collect()
     */
    function icollect($value = null)
    {
        return new Collection($value);
    }
}

if (! function_exists('es')) {
    /**
     * Encode HTML special characters in a string.
     *
     * @param DeferringDisplayableValue|Htmlable|string $value
     * @param bool $doubleEncode
     * @return string
     */
    function es($value, bool $doubleEncode = true): string
    {
        if ($value instanceof DeferringDisplayableValue) {
            $value = $value->resolveDisplayableValue();
        }

        if ($value instanceof Htmlable) {
            return $value->toHtml();
        }

        return htmlspecialchars($value, ENT_QUOTES, 'UTF-8', $doubleEncode);
    }
}

if (! function_exists('minifyHtml')) {
    function minifyHtml(string $value): string
    {
        $replace = [
            '/\>[^\S ]+/s' => '>',
            '/[^\S ]+\</s' => '<',
            '/([\t ])+/s' => ' ',
            '/^([\t ])+/m' => '',
            '/(?=<!--\s)([\s\S]*?)-->/' => '',
            '/([\t ])+$/m' => '',
            '~//[a-zA-Z0-9 ]+$~m' => '',
            '/[\r\n]+([\t ]?[\r\n]+)+/s' => "\n",
            '/\>[\r\n\t ]+\</s' => '><',
            '/}[\r\n\t ]+/s' => '}',
            '/}[\r\n\t ]+,[\r\n\t ]+/s' => '},',
            '/\)[\r\n\t ]?{[\r\n\t ]+/s' => '){',
            '/,[\r\n\t ]?{[\r\n\t ]+/s' => ',{',
            '/\),[\r\n\t ]+/s' => '),',
            '~([\r\n\t ])?([a-zA-Z0-9]+)=\"([a-zA-Z0-9_\\-]+)\"([\r\n\t ])?~s' => '$1$2=$3$4',
        ];

        /** @var string $result */
        $result = preg_replace(array_keys($replace), array_values($replace), $value);

        return $result;
    }
}

if (! function_exists('iClean')) {
    function iClean(string $dirty, string $type = 'textarea'): string
    {
        $configObject = HTMLPurifier_Config::createDefault();
        if (! config('purifier.finalize')) {
            $configObject->autoFinalize = false;
        }
        $defaultConfig = [];
        $defaultConfig['Core.Encoding'] = config('purifier.encoding');
        $defaultConfig['Cache.SerializerPath'] = config('purifier.cachePath');
        $defaultConfig['Cache.SerializerPermissions'] = config('purifier.cacheFileMode', 0755);
        $config = config('purifier.settings.default');
        if (! is_array($config)) {
            $config = [];
        }
        $config = $defaultConfig + $config;

        if ($type !== 'wysiwyg') {
            $config['AutoFormat.AutoParagraph'] = false;
        }

        $configObject->loadArray($config);
        // Add allowfullscreen on iframe
        $def = $configObject->getHTMLDefinition(true);
        if (null !== $def) {
            $def->addAttribute('iframe', 'allowfullscreen', 'Bool');
            $def->addAttribute('td', 'colwidth', 'Number');
            $def->addAttribute('th', 'colwidth', 'Number');
        }
        $purifier = new HTMLPurifier($configObject);

        return $purifier->purify($dirty);
    }
}

if (! function_exists('iUtf8LeftTrim')) {
    /**
     * @param string $str
     * @param string|false $charlist
     * @return string
     */
    function iUtf8LeftTrim(string $str, $charlist = false): string
    {
        if ($charlist === false) {
            return ltrim($str);
        }

        /** @var string $charlist */
        $charlist = preg_replace('!([\\\\\\-\\]\\[/^])!', '\\\${1}', $charlist);

        /** @var string $result */
        $result = preg_replace('/^['.$charlist.']+/u', '', $str);

        return $result;
    }
}

if (! function_exists('iUtf8RightTrim')) {
    /**
     * @param string $str
     * @param string|false $charlist
     * @return string
     */
    function iUtf8RightTrim($str, $charlist = false): string
    {
        if ($charlist === false) {
            return rtrim($str);
        }

        /** @var string $charlist */
        $charlist = preg_replace('!([\\\\\\-\\]\\[/^])!', '\\\${1}', $charlist);

        /** @var string $result */
        $result = preg_replace('/['.$charlist.']+$/u', '', $str);

        return $result;
    }
}

if (! function_exists('iUtf8Trim')) {
    /**
     * @param string $str
     * @param string|false $charlist
     * @return string
     */
    function iUtf8Trim(string $str, $charlist = false): string
    {
        if ($charlist === false) {
            return trim($str);
        }

        return iUtf8LeftTrim(iUtf8RightTrim($str, $charlist), $charlist);
    }
}

if (! function_exists('iTrim')) {
    /**
     * @param string $str
     * @param string|false $charlist
     * @return string
     */
    function iTrim(string $str, $charlist = false): string
    {
        if (empty($charlist) && $charlist !== false) {
            return $str;
        }

        if ($charlist === false) {
            return iUtf8Trim($str);
        }

        return iUtf8Trim($str, $charlist);
    }
}

if (! function_exists('iStrrpos')) {
    /**
     * @param string $str
     * @param string $search
     * @param int|false $offset
     * @return false|int
     */
    function iStrrpos(string $str, string $search, $offset = false)
    {
        if ($offset === false) {
            if (empty($str)) {
                return false;
            }

            return mb_strrpos($str, $search);
        } else {
            if (! is_int($offset)) {
                trigger_error('iStrrpos expects parameter 3 to be long', E_USER_WARNING);

                return false;
            }

            $str = mb_substr($str, $offset);

            if (false !== ($pos = mb_strrpos($str, $search))) {
                return $pos + $offset;
            }

            return false;
        }
    }
}
if (! function_exists('iHtmlTruncate')) {
    function htmlTruncate(string $text, int $length = 0, bool $noSplit = true, bool $allowHtml = true): string
    {
        if ($length === 1 && $text[0] === '<') {
            return '...';
        }

        if (! $allowHtml) {
            $text = str_replace('>', '> ', $text);
            $text = str_replace(['&nbsp;', '&#160;'], ' ', $text);
            /** @var string $text */
            $text = preg_replace('#\s+#mui', ' ', $text);
            $text = iTrim($text);

            $text = strip_tags($text);
            $text = html_entity_decode($text, ENT_QUOTES, 'UTF-8');

            $text = str_replace('&nbsp;', ' ', $text);
            /** @var string $text */
            $text = preg_replace('#\s+#mui', ' ', $text);
            $text = iTrim($text);
        }
        if ($length > 0 && Str::length($text) > $length) {
            $tmp = trim(Str::substr($text, 0, $length));

            if ($tmp[0] === '<' && strpos($tmp, '>') === false) {
                return '...';
            }

            if ($noSplit) {
                $offset = iStrrpos($tmp, ' ');
                $tmp = Str::substr($tmp, 0, $offset + 1);

                if ($offset === false && strlen($text) > $length) {
                    return '...';
                }

                if (Str::length($tmp) > $length - 3) {
                    $tmp = trim(Str::substr($tmp, 0, iStrrpos($tmp, ' ') ?: null));
                }
            }

            if ($allowHtml) {
                preg_match_all("#<([a-z][a-z0-9]*)\b.*?(?!/)>#i", $tmp, $result);
                $openedTags = $result[1];

                $openedTags = array_diff($openedTags, ['img', 'hr', 'br']);
                $openedTags = array_values($openedTags);

                preg_match_all("#</([a-z][a-z0-9]*)\b(?:[^>]*?)>#iU", $tmp, $result);
                $closedTags = $result[1];

                $numOpened = count($openedTags);
                if (count($closedTags) !== $numOpened) {
                    $openedTags = array_reverse($openedTags);
                    for ($i = 0; $i < $numOpened; $i++) {
                        if (! in_array($openedTags[$i], $closedTags)) {
                            $tmp .= '</'.$openedTags[$i].'>';
                        } else {
                            unset($closedTags[array_search($openedTags[$i], $closedTags)]);
                        }
                    }
                }

                if (iStrrpos($tmp, '<') > iStrrpos($tmp, '>')) {
                    $offset = iStrrpos($tmp, '<');
                    $tmp = iTrim(Str::substr($tmp, 0, $offset ?: null));
                }
            }

            if (strlen($text) > strlen($tmp)) {
                $text = trim($tmp).'...';
            }
        }

        $text = str_replace(' </', '</', $text);

        return str_replace(' ...', '...', $text);
    }
}

if (! function_exists('get_mail_logo')) {
    function get_mail_logo(): string
    {
        $appUrl = config('app.url').'/';

        if (file_exists(back_path('resources/assets/img/logo.png'))) {
            return $appUrl.'vendor/maecia/'.config('app.code').'-back/resources/assets/img/logo.png';
        }

        return $appUrl.'vendor/maecia/inside/resources/assets/img/logo.png';
    }
}

if (! function_exists('str')) {
    /**
     * @param  string|null  $string
     * @return Inside\Support\Stringable|mixed
     */
    function str(?string $string = null): mixed
    {
        if (func_num_args() === 0) {
            return new class() {
                public function __call(string $method, mixed $parameters): mixed
                {
                    return Str::$method(...$parameters);
                }

                public function __toString()
                {
                    return '';
                }
            };
        }

        return Str::of($string); // @phpstan-ignore-line
    }
}

if (! function_exists('warn_front')) {
    function warn_front(string $message, array $payload = []): void
    {
        if (is_null(env('LOG_FRONT_SLACK_WEBHOOK_URL'))) {
            return;
        }
        $request = app('request');
        if (preg_match('#^api/v1/content/([^/]+)$#', $request->path(), $matches)
        && $request->method() == 'GET') {
            // Inside content specific
            $payload['types'] = $matches[1];
            $payload['filters'] = ContentHelper::extractFiltersInputFromRequest($request);
            $payload['fields'] = ContentHelper::extractFieldsInputFromRequest($request);
        }
        $payload['path'] = $request->path();
        $payload['method'] = $request->method();
        $payload['request'] = $request->all();
        Log::channel('front_slack')->error($message, $payload);
    }
}
