<?php

namespace Inside\Validation;

use Egulias\EmailValidator\EmailValidator;
use Egulias\EmailValidator\Validation\DNSCheckValidation;
use Egulias\EmailValidator\Validation\MultipleValidationWithAnd;
use Egulias\EmailValidator\Validation\NoRFCWarningsValidation;
use Egulias\EmailValidator\Validation\RFCValidation;
use Egulias\EmailValidator\Validation\SpoofCheckValidation;
use Illuminate\Contracts\Translation\Translator as TranslatorContract;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\MessageBag;
use Illuminate\Translation\Translator;
use Illuminate\Validation\ValidationException;
use Illuminate\Validation\Validator as BaseValidator;
use Inside\Support\Str;
use Inside\Support\ValidatedInput;
use Symfony\Component\HttpFoundation\File\Exception\FileNotFoundException;
use Symfony\Component\HttpFoundation\File\File;

class Validator extends BaseValidator
{
    /**
     * The current placeholder for dots in rule keys.
     *
     * @var string
     */
    protected $dotPlaceholder;

    /**
     * @var Translator
     */
    protected $translator;

    /**
     * Validator constructor.
     *
     * @param TranslatorContract $translator
     * @param array $data
     * @param array $rules
     * @param array $messages
     * @param array $customAttributes
     */
    public function __construct(
        TranslatorContract $translator,
        array $data,
        array $rules,
        array $messages = [],
        array $customAttributes = []
    ) {
        parent::__construct($translator, $data, $rules, $messages, $customAttributes);
        $this->dotPlaceholder = Str::random();
        $this->sizeRules[] = 'Maxfilesize';
    }

    /**
     * parse data
     *
     * @param array $data
     * @return array
     */
    public function parseData(array $data)
    {
        $newData = [];

        foreach ($data as $key => $value) {
            if (is_array($value)) {
                $value = $this->parseData($value);
            }

            $key = str_replace(
                ['.', '*'],
                [$this->dotPlaceholder, '__asterisk__'],
                $key
            );

            $newData[$key] = $value;
        }

        return $newData;
    }

    /**
     * Check with rules if data is valid
     *
     * @return array
     * @throws ValidationException
     */
    public function validate(): array
    {
        if ($this->fails()) {
            throw new ValidationException($this);
        }

        return $this->validated();
    }

    /**
     * Get validated attributes
     *
     * @return array
     * @throws ValidationException
     */
    public function validated(): array
    {
        if ($this->invalid()) {
            throw new ValidationException($this);
        }

        $results = [];

        $missingValue = Str::random(10);

        foreach (array_keys($this->getRules()) as $key) {
            $value = data_get($this->getData(), $key, $missingValue);

            if ($value !== $missingValue) {
                Arr::set($results, $key, $value);
            }
        }

        return $this->replacePlaceholders($results);
    }

    /**
     * Replace placeholder with dot and asterisk in $data keys
     *
     * @param array $data
     * @return array
     */
    protected function replacePlaceholders(array $data): array
    {
        $originalData = [];

        foreach ($data as $key => $value) {
            $originalData[$this->replacePlaceholderInString($key)] =
                is_array($value) ? $this->replacePlaceholders($value) : $value;
        }

        return $originalData;
    }

    /**
     * Replace placeholder with dot and asterisk in $value
     *
     * @param string $value
     * @return string
     */
    protected function replacePlaceholderInString(string $value): string
    {
        /** @var string $result */
        $result = str_replace(
            [$this->dotPlaceholder, '__asterisk__'],
            ['.', '*'],
            $value
        );

        return $result;
    }

    /**
     * Add a failed validation error and message for rule $rule
     *
     * @param string $attribute
     * @param string $rule
     * @param array $parameters
     */
    public function addFailure($attribute, $rule, $parameters = [])
    {
        /** @var MessageBag|null $messages */
        $messages = $this->messages;
        if (! $messages) {
            $this->passes();
        }

        $attribute = str_replace(
            [$this->dotPlaceholder, '__asterisk__'],
            ['.', '*'],
            $attribute
        );

        $this->messages->add(
            $attribute,
            $this->makeReplacements(
                $this->getMessage($attribute, $rule),
                $attribute,
                $rule,
                $parameters
            )
        );

        $this->failedRules[$attribute][$rule] = $parameters;
    }

    /**
     * validate Mimes
     *
     * @param string $attribute
     * @param mixed $value
     * @param array $parameters
     * @return bool
     */
    public function validateMimes($attribute, $value, $parameters)
    {
        if (is_string($value)) {
            try {
                /** @phpstan-ignore-next-line  */
                $value = new File(Storage::disk('local')->path($value));
            } catch (FileNotFoundException $e) {
                return false;
            }
        }

        return parent::validateMimes($attribute, $value, $parameters);
    }

    /**
     * Validate mime types
     *
     * @param string $attribute
     * @param mixed $value
     * @param array $parameters
     * @return bool
     */
    public function validateMimetypes($attribute, $value, $parameters)
    {
        if (is_string($value)) {
            try {
                /** @phpstan-ignore-next-line  */
                $value = new File(Storage::disk('local')->path($value));
            } catch (FileNotFoundException $e) {
                return false;
            }
        }

        if (! $this->isValidFileInstance($value)) {
            return false;
        }

        if ($this->shouldBlockPhpUpload($value, $parameters)) {
            return false;
        }

        $mimeType = strtolower($value->getMimeType());

        $validated = $value->getPath() !== '' &&
            (in_array($mimeType, $parameters) ||
                in_array(explode('/', $mimeType)[0].'/*', $parameters));

        if (! $validated) {
            Log::debug(sprintf('[Mimetype validation] got mimetype %s for extension %s, expected %s', $value->getMimeType(), $value->getExtension(), implode(',', $parameters)));
        }

        return $validated;
    }

    /**
     * @param string $attribute
     * @param mixed $value
     * @param array $parameters
     * @return bool
     */
    public function validateMaxfilesize(string $attribute, $value, array $parameters): bool
    {
        if (is_string($value)) {
            try {
                /** @phpstan-ignore-next-line  */
                $value = new File(Storage::disk('local')->path($value));
            } catch (FileNotFoundException $e) {
                return false;
            }
        }

        return $value instanceof File && $this->validateMax($attribute, $value, $parameters);
    }

    /**
     * @param string $attribute
     * @param mixed $value
     * @param array $parameters
     * @return bool
     */
    public function validateDimensions($attribute, $value, $parameters): bool
    {
        if (is_string($value)) {
            try {
                /** @phpstan-ignore-next-line  */
                $value = new File(Storage::disk('local')->path($value));
            } catch (FileNotFoundException $e) {
                return false;
            }
        }

        return parent::validateDimensions($attribute, $value, $parameters);
    }

    /**
     * @param string $attribute
     * @param mixed $value
     * @param array $parameters
     * @return bool
     */
    public function validateTimestamp(string $attribute, $value, array $parameters): bool
    {
        return ((string) (int) $value === $value)
            && ($value <= PHP_INT_MAX)
            && ($value >= ~PHP_INT_MAX);
    }

    /**
     * Validate an uuid
     *
     * @param string $attribute
     * @param mixed $value
     * @return bool
     */
    public function validateUuid($attribute, $value): bool
    {
        return Str::isUuid($value);
    }

    /**
     * @param string $attribute
     * @param mixed $value
     * @return bool
     */
    public function validateTimezone($attribute, $value): bool
    {
        return in_array($value, timezone_identifiers_list(), true);
    }

    /**
     * @param $attribute
     * @param $value
     * @return bool
     */
    public function validatePhone($attribute, $value): bool
    {
        return true;
    }

    /**
     * @param string $attribute
     * @param mixed $value
     * @param array $parameters
     * @return bool
     */
    public function validateEmailpp(string $attribute, $value, array $parameters): bool
    {
        if (! is_string($value) && ! (is_object($value) && method_exists($value, '__toString'))) {
            return false;
        }

        $validations = collect($parameters)->unique()->map(
            function ($validation) {
                if ($validation === 'rfc') {
                    return new RFCValidation();
                } elseif ($validation === 'strict') {
                    return new NoRFCWarningsValidation();
                } elseif ($validation === 'dns') {
                    return new DNSCheckValidation();
                } elseif ($validation === 'spoof') {
                    return new SpoofCheckValidation();
                } elseif ($validation === 'filter') {
                    return new FilterEmailValidation();
                } elseif ($validation === 'filter_unicode') {
                    return FilterEmailValidation::unicode();
                } elseif (is_string($validation) && class_exists($validation)) {
                    return $this->container->make($validation);
                }
            }
        )->values()->all() ?: [new RFCValidation()];

        return (new EmailValidator())->isValid((string) $value, new MultipleValidationWithAnd($validations));
    }

    /**
     * validate a local
     *
     * @param string $attribute
     * @param mixed $value
     * @return bool
     */
    public function validateLocale(string $attribute, $value): bool
    {
        return in_array($value, list_languages(), true);
    }

    /**
     * @param string $attribute
     * @param mixed $value
     * @param mixed $parameters
     * @return bool
     */
    public function validateExcludeUnless($attribute, $value, $parameters)
    {
        $this->requireParameterCount(2, $parameters, 'exclude_unless');

        [$values, $other] = $this->prepareValuesAndOther($parameters);

        return in_array($other, $values);
    }

    protected function prepareValuesAndOther(mixed $parameters): array
    {
        $other = Arr::get($this->data, $parameters[0]);

        $values = array_slice($parameters, 1);

        if (is_bool($other)) {
            $values = $this->convertValuesToBoolean($values);
        } elseif (is_null($other)) {
            $values = $this->convertValuesToNull($values);
        }

        return [$values, $other];
    }

    protected function convertValuesToNull(mixed $values): array
    {
        return array_map(
            function ($value) {
                return Str::lower($value) === 'null' ? null : $value;
            },
            $values
        );
    }

    /**
     * get Custom Message
     *
     * @param string $key
     * @return string
     */
    protected function getCustomMessageFromTranslator($key): string
    {
        /** @var string $message */
        /** @phpstan-ignore-next-line  */
        $message = $this->translator->getFromJson($key);

        if ($message !== $key) {
            return $message;
        }

        /** @var string $shortKey */
        $shortKey = preg_replace(
            '/^validation\.custom\./',
            'validation.',
            $key
        );

        /** @var string $shortMessage */
        /** @phpstan-ignore-next-line  */
        $shortMessage = $this->translator->getFromJson($shortKey);

        if ($shortMessage !== $shortKey) {
            return $message;
        }

        return parent::getCustomMessageFromTranslator($key);
    }

    protected function getSizeMessage($attribute, $rule)
    {
        $lowerRule = Str::snake($rule);

        $type = $this->getAttributeType($attribute);

        $key = "validation.{$lowerRule}.{$type}";

        /** @var string $message */
        /** @phpstan-ignore-next-line  */
        $message = $this->translator->getFromJson($key);

        if ($message !== $key) {
            return $message;
        }

        return parent::getSizeMessage($attribute, $rule);
    }

    protected function getAttributeType($attribute)
    {
        $value = $this->getValue($attribute);
        if (is_string($value)) {
            try {
                /** @phpstan-ignore-next-line  */
                $value = new File(Storage::disk('local')->path($value));
            } catch (FileNotFoundException $e) {
            }
        }

        if ($value instanceof File) {
            return 'file';
        } else {
            return parent::getAttributeType($attribute);
        }
    }

    /**
     * $value must be an image!
     *
     * @param string $attribute
     * @param mixed $value
     * @return bool
     */
    public function validateImage($attribute, $value)
    {
        return $this->validateMimes($attribute, $value, ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'svg', 'webp']);
    }

    /**
     * Get safe inputs
     *
     * @param array|null $keys
     * @return array|ValidatedInput
     */
    public function safe(array $keys = null)
    {
        return is_array($keys)
            ? (new ValidatedInput($this->validated()))->only($keys)
            : new ValidatedInput($this->validated());
    }
}
