<?php

namespace Inside\Support;

use Closure;
use Illuminate\Support\Collection;
use Illuminate\Support\Traits\Macroable;
use Symfony\Component\VarDumper\VarDumper;

class Stringable
{
    use Macroable;

    /**
     * The underlying string value.
     *
     * @var string
     */
    protected $value;

    /**
     * Create a new instance of the class.
     *
     * @param string $value
     */
    public function __construct($value = '')
    {
        $this->value = (string) $value;
    }

    /**
     * Return the remainder of a string after the first occurrence of a given value.
     *
     * @param string $search
     * @return static
     */
    public function after(string $search): self
    {
        return new static(Str::after($this->value, $search));
    }

    /**
     * Return the remainder of a string after the last occurrence of a given value.
     *
     * @param string $search
     * @return static
     */
    public function afterLast(string $search): self
    {
        return new static(Str::afterLast($this->value, $search));
    }

    /**
     * Append the given values to the string.
     *
     * @param string $values
     * @return static
     */
    public function append(...$values): self
    {
        return new static($this->value.implode('', array_values($values)));
    }

    /**
     * Transliterate a UTF-8 value to ASCII.
     *
     * @param string $language
     * @return static
     */
    public function ascii(string $language = 'en'): self
    {
        return new static(Str::ascii($this->value, $language));
    }

    /**
     * Get the trailing name component of the path.
     *
     * @param string $suffix
     * @return static
     */
    public function basename(string $suffix = ''): self
    {
        return new static(basename($this->value, $suffix));
    }

    /**
     * Get the portion of a string before the first occurrence of a given value.
     *
     * @param string $search
     * @return static
     */
    public function before(string $search): self
    {
        return new static(Str::before($this->value, $search));
    }

    /**
     * Get the portion of a string before the last occurrence of a given value.
     *
     * @param string $search
     * @return static
     */
    public function beforeLast(string $search): self
    {
        return new static(Str::beforeLast($this->value, $search));
    }

    /**
     * Get the portion of a string between two given values.
     *
     * @param string $from
     * @param string $to
     * @return static
     */
    public function between(string $from, string $to): self
    {
        return new static(Str::between($this->value, $from, $to));
    }

    /**
     * Convert a value to camel case.
     *
     * @return static
     */
    public function camel(): self
    {
        return new static(Str::camel($this->value));
    }

    /**
     * Determine if a given string contains a given substring.
     *
     * @param string|array $needles
     * @return bool
     */
    public function contains($needles): bool
    {
        return Str::contains($this->value, $needles);
    }

    /**
     * Determine if a given string contains all array values.
     *
     * @param array $needles
     * @return bool
     */
    public function containsAll(array $needles): bool
    {
        return Str::containsAll($this->value, $needles);
    }

    /**
     * Get the parent directory's path.
     *
     * @param int<1, max> $levels
     * @return static
     */
    public function dirname(int $levels = 1): self
    {
        return new static(dirname($this->value, $levels));
    }

    /**
     * Determine if a given string ends with a given substring.
     *
     * @param string|array $needles
     * @return bool
     */
    public function endsWith($needles): bool
    {
        return Str::endsWith($this->value, $needles);
    }

    /**
     * Determine if the string is an exact match with the given value.
     *
     * @param string $value
     * @return bool
     */
    public function exactly(string $value): bool
    {
        return $this->value === $value;
    }

    /**
     * Explode the string into an array.
     *
     * @param string $delimiter
     * @param int $limit
     * @return Collection
     */
    public function explode(string $delimiter, int $limit = PHP_INT_MAX): Collection
    {
        if (empty($delimiter)) {
            return collect();
        }

        return collect(explode($delimiter, $this->value, $limit));
    }

    /**
     * Split a string using a regular expression.
     *
     * @param string $pattern
     * @param int $limit
     * @param int $flags
     * @return Collection
     */
    public function split(string $pattern, int $limit = -1, int $flags = 0): Collection
    {
        $segments = preg_split($pattern, $this->value, $limit, $flags);

        return ! empty($segments) ? collect($segments) : collect();
    }

    /**
     * Cap a string with a single instance of a given value.
     *
     * @param string $cap
     * @return static
     */
    public function finish(string $cap): self
    {
        return new static(Str::finish($this->value, $cap));
    }

    /**
     * Determine if a given string matches a given pattern.
     *
     * @param string|array $pattern
     * @return bool
     */
    public function is($pattern): bool
    {
        return Str::is($pattern, $this->value);
    }

    /**
     * Determine if a given string is 7 bit ASCII.
     *
     * @return bool
     */
    public function isAscii(): bool
    {
        return Str::isAscii($this->value);
    }

    /**
     * Determine if the given string is empty.
     *
     * @return bool
     */
    public function isEmpty(): bool
    {
        return $this->value === '';
    }

    /**
     * Determine if the given string is not empty.
     *
     * @return bool
     */
    public function isNotEmpty(): bool
    {
        return ! $this->isEmpty();
    }

    /**
     * Convert a string to kebab case.
     *
     * @return static
     */
    public function kebab(): self
    {
        return new static(Str::kebab($this->value));
    }

    /**
     * Return the length of the given string.
     *
     * @param string|null $encoding
     * @return int
     */
    public function length(?string $encoding = null): int
    {
        return Str::length($this->value, $encoding);
    }

    /**
     * Limit the number of characters in a string.
     *
     * @param int $limit
     * @param string $end
     * @return static
     */
    public function limit(int $limit = 100, string $end = '...'): self
    {
        return new static(Str::limit($this->value, $limit, $end));
    }

    /**
     * Convert the given string to lower-case.
     *
     * @return static
     */
    public function lower(): self
    {
        return new static(Str::lower($this->value));
    }

    /**
     * Get the string matching the given pattern.
     *
     * @param string $pattern
     * @return static|null
     */
    public function match(string $pattern): ?self
    {
        preg_match($pattern, $this->value, $matches);

        if (! $matches) {
            return new static();
        }

        return new static($matches[1] ?? $matches[0]);
    }

    /**
     * Get the string matching the given pattern.
     *
     * @param string $pattern
     * @return Collection
     */
    public function matchAll(string $pattern): Collection
    {
        preg_match_all($pattern, $this->value, $matches);

        if (empty($matches[0])) {
            return collect();
        }

        return collect($matches[1] ?? $matches[0]);
    }

    /**
     * Pad both sides of the string with another.
     *
     * @param int $length
     * @param string $pad
     * @return static
     */
    public function padBoth(int $length, string $pad = ' '): self
    {
        return new static(Str::padBoth($this->value, $length, $pad));
    }

    /**
     * Pad the left side of the string with another.
     *
     * @param int $length
     * @param string $pad
     * @return static
     */
    public function padLeft(int $length, string $pad = ' '): self
    {
        return new static(Str::padLeft($this->value, $length, $pad));
    }

    /**
     * Pad the right side of the string with another.
     *
     * @param int $length
     * @param string $pad
     * @return static
     */
    public function padRight(int $length, string $pad = ' '): self
    {
        return new static(Str::padRight($this->value, $length, $pad));
    }

    /**
     * Parse a Class@method style callback into class and method.
     *
     * @param string|null $default
     * @return array
     */
    public function parseCallback(?string $default = null): array
    {
        return Str::parseCallback($this->value, $default);
    }

    /**
     * Get the plural form of an English word.
     *
     * @param int $count
     * @return static
     */
    public function plural(int $count = 2): self
    {
        return new static(Str::plural($this->value, $count));
    }

    /**
     * Pluralize the last word of an English, studly caps case string.
     *
     * @param int $count
     * @return static
     */
    public function pluralStudly(int $count = 2): self
    {
        return new static(Str::pluralStudly($this->value, $count));
    }

    /**
     * Prepend the given values to the string.
     *
     * @param string $values
     * @return static
     */
    public function prepend(...$values): self
    {
        return new static(implode('', $values).$this->value);
    }

    /**
     * Replace the given value in the given string.
     *
     * @param string|string[] $search
     * @param string|string[] $replace
     * @return static
     */
    public function replace($search, $replace): self
    {
        return new static(str_replace($search, $replace, $this->value));
    }

    /**
     * Replace a given value in the string sequentially with an array.
     *
     * @param string $search
     * @param array $replace
     * @return static
     */
    public function replaceArray($search, array $replace): self
    {
        return new static(Str::replaceArray($search, $replace, $this->value));
    }

    /**
     * Replace the first occurrence of a given value in the string.
     *
     * @param string $search
     * @param string $replace
     * @return static
     */
    public function replaceFirst(string $search, string $replace): self
    {
        return new static(Str::replaceFirst($search, $replace, $this->value));
    }

    /**
     * Replace the last occurrence of a given value in the string.
     *
     * @param string $search
     * @param string $replace
     * @return static
     */
    public function replaceLast(string $search, string $replace): self
    {
        return new static(Str::replaceLast($search, $replace, $this->value));
    }

    /**
     * Replace the patterns matching the given regular expression.
     *
     * @param string $pattern
     * @param Closure|string $replace
     * @param int $limit
     * @return static
     */
    public function replaceMatches(string $pattern, $replace, int $limit = -1): self
    {
        if ($replace instanceof Closure) {
            return new static(preg_replace_callback($pattern, $replace, $this->value, $limit));
        }

        return new static(preg_replace($pattern, $replace, $this->value, $limit));
    }

    /**
     * Begin a string with a single instance of a given value.
     *
     * @param string $prefix
     * @return static
     */
    public function start(string $prefix): self
    {
        return new static(Str::start($this->value, $prefix));
    }

    /**
     * Convert the given string to upper-case.
     *
     * @return static
     */
    public function upper(): self
    {
        return new static(Str::upper($this->value));
    }

    /**
     * Convert the given string to title case.
     *
     * @return static
     */
    public function title(): self
    {
        return new static(Str::title($this->value));
    }

    /**
     * Get the singular form of an English word.
     *
     * @return static
     */
    public function singular(): self
    {
        return new static(Str::singular($this->value));
    }

    /**
     * Generate a URL friendly "slug" from a given string.
     *
     * @param string $separator
     * @param string|null $language
     * @return static
     */
    public function slug(string $separator = '-', ?string $language = 'en'): self
    {
        return new static(Str::slug($this->value, $separator, $language ?? 'en'));
    }

    /**
     * Convert a string to snake case.
     *
     * @param string $delimiter
     * @return static
     */
    public function snake(string $delimiter = '_'): self
    {
        return new static(Str::snake($this->value, $delimiter));
    }

    /**
     * Determine if a given string starts with a given substring.
     *
     * @param string|array $needles
     * @return bool
     */
    public function startsWith($needles): bool
    {
        return Str::startsWith($this->value, $needles);
    }

    /**
     * Convert a value to studly caps case.
     *
     * @return static
     */
    public function studly(): self
    {
        return new static(Str::studly($this->value));
    }

    /**
     * Returns the portion of string specified by the start and length parameters.
     *
     * @param int $start
     * @param int|null $length
     * @return static
     */
    public function substr(int $start, ?int $length = null): self
    {
        return new static(Str::substr($this->value, $start, $length));
    }

    /**
     * Returns the number of substring occurrences.
     *
     * @param string $needle
     * @param int|null $offset
     * @param int|null $length
     * @return int
     */
    public function substrCount(string $needle, ?int $offset = null, ?int $length = null): int
    {
        return Str::substrCount($this->value, $needle, (int) $offset, $length);
    }

    /**
     * Trim the string of the given characters.
     *
     * @param string|null $characters
     * @return static
     */
    public function trim(?string $characters = null): self
    {
        return new static(trim(...array_merge([$this->value], func_get_args())));
    }

    /**
     * Left trim the string of the given characters.
     *
     * @param string|null $characters
     * @return static
     */
    public function ltrim(?string $characters = null): self
    {
        return new static(ltrim(...array_merge([$this->value], func_get_args())));
    }

    /**
     * Right trim the string of the given characters.
     *
     * @param string|null $characters
     * @return static
     */
    public function rtrim(?string $characters = null): self
    {
        return new static(rtrim(...array_merge([$this->value], func_get_args())));
    }

    /**
     * Make a string's first character uppercase.
     *
     * @return static
     */
    public function ucfirst(): self
    {
        return new static(Str::ucfirst($this->value));
    }

    /**
     * Apply the callback's string changes if the given "value" is true.
     *
     * @param mixed $value
     * @param callable $callback
     * @param callable|null $default
     * @return mixed|$this
     */
    public function when($value, callable $callback, ?callable $default = null)
    {
        if ($value) {
            return $callback($this, $value) ?: $this;
        } elseif ($default) {
            return $default($this, $value) ?: $this;
        }

        return $this;
    }

    /**
     * Execute the given callback if the string is empty.
     *
     * @param callable $callback
     * @return static
     */
    public function whenEmpty(callable $callback): self
    {
        if ($this->isEmpty()) {
            $result = $callback($this);

            return is_null($result) ? $this : $result;
        }

        return $this;
    }

    /**
     * Limit the number of words in a string.
     *
     * @param int $words
     * @param string $end
     * @return static
     */
    public function words(int $words = 100, string $end = '...'): self
    {
        return new static(Str::words($this->value, $words, $end));
    }

    public function toString(): string
    {
        return $this->value;
    }

    /**
     * Dump the string.
     *
     * @return $this
     */
    public function dump(): self
    {
        VarDumper::dump($this->value);

        return $this;
    }

    /**
     * Dump the string and end the script.
     *
     * @return void
     */
    public function dd(): void
    {
        $this->dump();

        exit(1);
    }

    /**
     * Proxy dynamic properties onto methods.
     *
     * @param string $key
     * @return mixed
     */
    public function __get($key)
    {
        return $this->{$key}();
    }

    /**
     * Get the raw string value.
     *
     * @return string
     */
    public function __toString()
    {
        return (string) $this->value;
    }
}
