<?php

namespace Inside\Support;

use Closure;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Str;

class RouteFormatter
{
    protected array $headers = ['Domain', 'Method', 'URI', 'Name', 'Action', 'Middleware'];

    protected bool $compact;

    protected array $compactColumns = ['method', 'uri', 'action'];

    protected ?string $name;

    protected ?string $path;

    protected ?string $method;

    protected ?string $sort;

    protected bool $reverse;

    protected array $columns;

    protected ?string $exceptPath;

    public function getRoutes(): array
    {
        $routes = collect(Route::getRoutes())->map(
            function ($route) {
                return $this->getRouteInformation((array) $route);
            }
        )->filter()->all();

        if (! is_null($this->sort) && $this->sort !== 'precedence') {
            $routes = $this->sortRoutes($this->sort, $routes);
        }
        if ($this->reverse) {
            $routes = array_reverse($routes);
        }

        return $this->pluckColumns($routes);
    }

    public function getRouteHeaders(): array
    {
        return Arr::only($this->headers, array_keys($this->getColumns()));
    }

    public function getName(): ?string
    {
        return $this->name;
    }

    public function setName(?string $name): self
    {
        $this->name = $name;

        return $this;
    }

    public function getPath(): ?string
    {
        return $this->path;
    }

    public function setPath(?string $path): self
    {
        $this->path = $path;

        return $this;
    }

    public function getExceptPath(): ?string
    {
        return $this->exceptPath;
    }

    public function setExceptPath(?string $exceptPath): self
    {
        $this->exceptPath = $exceptPath;

        return $this;
    }

    public function getMethod(): ?string
    {
        return $this->method;
    }

    public function setMethod(?string $method): self
    {
        $this->method = $method;

        return $this;
    }

    public function getSort(): ?string
    {
        return $this->sort;
    }

    public function setSort(string $sort): self
    {
        $this->sort = $sort;

        return $this;
    }

    public function isReverse(): bool
    {
        return $this->reverse;
    }

    public function setReverse(bool $reverse): self
    {
        $this->reverse = $reverse;

        return $this;
    }

    public function isCompact(): bool
    {
        return $this->compact;
    }

    public function setCompact(bool $compact): self
    {
        $this->compact = $compact;

        return $this;
    }

    public function getHeaders(): array
    {
        return Arr::only($this->headers, array_keys($this->getColumns()));
    }

    public function reset(): self
    {
        $this->reverse = $this->compact = false;
        $this->method = $this->name = $this->path = $this->sort = null;

        return $this;
    }

    protected function getRouteInformation(array $information): array
    {
        return $this->filterRoute(
            [
                'domain' => $this->getRouteDomain($information),
                'method' => implode('|', $this->getRouteMethods($information) ?? []),
                'uri' => $this->getRouteUri($information),
                'name' => $this->getRouteName($information),
                'action' => ltrim($this->getRouteActionName($information) ?? '', '\\'),
                'middleware' => $this->getRouteMiddleware($information),
            ]
        );
    }

    /**
     * pluck columns for each route
     *
     * @param array $routes
     * @return array
     */
    protected function pluckColumns(array $routes): array
    {
        return array_map(
            function ($route) {
                return Arr::only($route, $this->getColumns());
            },
            $routes
        );
    }

    /**
     * @param array $columns
     * @return $this
     */
    public function setColumns(array $columns): self
    {
        $this->columns = $columns;

        return $this;
    }

    /**
     * get route formatter columns
     *
     * @return array
     */
    protected function getColumns(): array
    {
        $availableColumns = array_map('strtolower', $this->headers);

        if ($this->compact) {
            return array_intersect($availableColumns, $this->compactColumns);
        }

        if (! empty($this->columns)) {
            return array_intersect($availableColumns, $this->parseColumns($this->columns));
        }

        return $availableColumns;
    }

    /**
     * Parse route columns
     *
     * @param array $columns
     * @return array
     */
    protected function parseColumns(array $columns): array
    {
        $results = [];

        foreach ($columns as $column) {
            if (Str::contains($column, ',')) {
                $results = array_merge($results, explode(',', $column));
            } else {
                $results[] = $column;
            }
        }

        return array_map('strtolower', $results);
    }

    protected function filterRoute(array $route): array
    {
        if (
            (! is_null($this->name) && ! Str::contains($route['name'], $this->name))
            || ! is_null($this->path) && ! Str::contains($route['uri'], $this->path)
            || ! is_null($this->method) && ! Str::contains($route['method'], strtoupper($this->method))
        ) {
            return [];
        }

        if (! is_null($this->exceptPath)) {
            foreach (explode(',', $this->exceptPath) as $path) {
                if (Str::contains($route['uri'], $path)) {
                    return [];
                }
            }
        }

        return $route;
    }

    protected function getRouteDomain(array $information): ?string
    {
        if (
            array_key_exists('action', $information) &&
            is_array($information['action']) &&
            array_key_exists('domain', $information['action']) &&
            is_string($information['action']['domain'])
        ) {
            return str_replace(['http://', 'https://'], '', $information['action']['domain']);
        }

        return null;
    }

    protected function getRouteMethods(array $information): ?array
    {
        if (array_key_exists('method', $information)) {
            if (is_string($information['method'])) {
                return [$information['method']];
            } elseif (is_array($information['method'])) {
                return array_filter(
                    $information['method'],
                    function ($method) {
                        return is_string($method);
                    }
                );
            }
        }

        return null;
    }

    protected function getRouteUri(array $information): ?string
    {
        if (array_key_exists('uri', $information) && is_string($information['uri'])) {
            return $information['uri'];
        }

        return null;
    }

    protected function getRouteName(array $information): ?string
    {
        if (
            array_key_exists('action', $information)
            && is_array($information['action']) &&
            array_key_exists('as', $information['action']) &&
            is_string($information['action']['as'])
        ) {
            return $information['action']['as'];
        }

        return null;
    }

    protected function getRouteActionName(array $information): ?string
    {
        if (
            array_key_exists('action', $information) &&
            is_array($information['action']) &&
            array_key_exists('uses', $information['action']) &&
            is_string($information['action']['uses'])
        ) {
            return $information['action']['uses'];
        }

        return 'Closure';
    }

    protected function getRouteMiddleware(array $information): ?string
    {
        if (
            array_key_exists('action', $information) &&
            is_array($information['action']) &&
            array_key_exists('middleware', $information['action']) &&
            is_array($information['action']['middleware'])
        ) {
            return collect($information['action']['middleware'])->map(
                function ($middleware) {
                    return $middleware instanceof Closure ? 'Closure' : $middleware;
                }
            )->implode(',');
        }

        return null;
    }

    protected function sortRoutes(?string $sort, array $routes): array
    {
        if ($sort === null) {
            return $routes;
        }

        return Arr::sort(
            $routes,
            function ($route) use ($sort) {
                return $route[$sort];
            }
        );
    }
}
