<?php

namespace Inside\Support\Requirements;

use Closure;

class RequirementNameResolver
{
    /**
     * Resolve the requirement name to a class name(s) preserving passed parameters.
     *
     * @param  Closure|string  $name
     * @param  array  $requirementsMap
     * @param  array  $requirementGroups
     * @return Closure|string|array
     */
    public static function resolve($name, array $requirementsMap, array $requirementGroups)
    {
        if ($name instanceof Closure) {
            return $name;
        }

        if (isset($requirementsMap[$name]) && $requirementsMap[$name] instanceof Closure) {
            return $requirementsMap[$name];
        }

        if (isset($requirementGroups[$name])) {
            return static::parseRequirementGroup($name, $requirementsMap, $requirementGroups);
        }

        [$name, $parameters] = array_pad(explode(':', $name, 2), 2, null);

        return ($requirementsMap[$name] ?? $name).(! is_null($parameters) ? ':'.$parameters : '');
    }

    /**
     * Parse the requirement group and format it for usage.
     *
     * @param  string  $name
     * @param  array  $requirementsMap
     * @param  array  $requirementGroups
     * @return array
     */
    protected static function parseRequirementGroup(
        string $name,
        array $requirementsMap,
        array $requirementGroups
    ): array {
        $results = [];

        foreach ($requirementGroups[$name] as $requirement) {
            if (isset($requirementGroups[$requirement])) {
                $results = array_merge($results, static::parseRequirementGroup(
                    $requirement,
                    $requirementsMap,
                    $requirementGroups
                ));

                continue;
            }

            [$requirement, $parameters] = array_pad(
                explode(':', $requirement, 2),
                2,
                null
            );
            if (isset($requirementsMap[$requirement])) {
                $requirement = $requirementsMap[$requirement];
            }

            $results[] = $requirement.($parameters ? ':'.$parameters : '');
        }

        return $results;
    }
}
