<?php

namespace Inside\Support\Http\Client;

use Closure;
use Exception;
use GuzzleHttp\Client;
use GuzzleHttp\Cookie\CookieJar;
use GuzzleHttp\Exception\ConnectException;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\Exception\TransferException;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Promise\PromiseInterface;
use GuzzleHttp\TransferStats;
use Illuminate\Support\Collection;
use Illuminate\Support\Traits\Macroable;
use Inside\Support\Http\Client\Events\ConnectionFailed;
use Inside\Support\Http\Client\Events\RequestSending;
use Inside\Support\Http\Client\Events\ResponseReceived;
use Inside\Support\Str;
use Psr\Http\Message\MessageInterface;
use Psr\Http\Message\RequestInterface;
use Symfony\Component\VarDumper\VarDumper;

class PendingRequest
{
    use Macroable;

    /**
     * The factory instance.
     *
     * @var Factory|null
     */
    protected $factory;

    /**
     * The Guzzle client instance.
     *
     * @var Client|null
     */
    protected $client = null;

    /**
     * The base URL for the request.
     *
     * @var string
     */
    protected $baseUrl = '';

    /**
     * The request body format.
     *
     * @var string
     */
    protected $bodyFormat;

    /**
     * The raw body for the request.
     *
     * @var string|null
     */
    protected $pendingBody;

    /**
     * The pending files for the request.
     *
     * @var array
     */
    protected $pendingFiles = [];

    /**
     * The request cookies.
     *
     * @var array
     */
    protected $cookies;

    /**
     * The transfer stats for the request.
     *
     * @var TransferStats
     */
    protected $transferStats;

    /**
     * The request options.
     *
     * @var array
     */
    protected $options = [];

    /**
     * The number of times to try the request.
     *
     * @var int
     */
    protected $tries = 1;

    /**
     * The number of milliseconds to wait between retries.
     *
     * @var int
     */
    protected $retryDelay = 100;

    /**
     * The callback that will determine if the request should be retried.
     *
     * @var callable|null
     */
    protected $retryWhenCallback = null;

    /**
     * The callbacks that should execute before the request is sent.
     *
     * @var Collection
     */
    protected $beforeSendingCallbacks;

    /**
     * The stub callables that will handle requests.
     *
     * @var Collection|null
     */
    protected $stubCallbacks;

    /**
     * The middleware callables added by users that will handle requests.
     *
     * @var Collection
     */
    protected $middleware;

    /**
     * Whether the requests should be asynchronous.
     *
     * @var bool
     */
    protected $async = false;

    /**
     * The pending request promise.
     *
     * @var PromiseInterface
     */
    protected $promise;

    /**
     * The sent request object, if a request has been made.
     *
     * @var Request|null
     */
    protected $request;

    /**
     * Create a new HTTP Client instance.
     *
     * @param Factory|null $factory
     * @return void
     */
    public function __construct(Factory $factory = null)
    {
        $this->factory = $factory;
        $this->middleware = new Collection();

        $this->asJson();

        $this->options = [
            'http_errors' => false,
        ];

        $this->beforeSendingCallbacks = collect([
            function (Request $request, array $options, self $pendingRequest) {
                $pendingRequest->request = $request;
                $pendingRequest->cookies = $options['cookies'];

                $pendingRequest->dispatchRequestSendingEvent();
            },
        ]);
    }

    /**
     * Set the base URL for the pending request.
     *
     * @param string $url
     * @return $this
     */
    public function baseUrl(string $url): self
    {
        $this->baseUrl = $url;

        return $this;
    }

    /**
     * Attach a raw body to the request.
     *
     * @param string $content
     * @param string $contentType
     * @return $this
     */
    public function withBody(string $content, string $contentType): self
    {
        $this->bodyFormat('body');

        $this->pendingBody = $content;

        $this->contentType($contentType);

        return $this;
    }

    /**
     * Indicate the request contains JSON.
     *
     * @return $this
     */
    public function asJson(): self
    {
        return $this->bodyFormat('json')->contentType('application/json');
    }

    /**
     * Indicate the request contains form parameters.
     *
     * @return $this
     */
    public function asForm(): self
    {
        return $this->bodyFormat('form_params')->contentType('application/x-www-form-urlencoded');
    }

    /**
     * Attach a file to the request.
     *
     * @param string|array $name
     * @param string $contents
     * @param string|null $filename
     * @param array $headers
     * @return $this
     */
    public function attach($name, string $contents = '', string $filename = null, array $headers = []): self
    {
        if (is_array($name)) {
            foreach ($name as $file) {
                $this->attach(...$file);
            }

            return $this;
        }

        $this->asMultipart();

        $this->pendingFiles[] = array_filter([
            'name' => $name,
            'contents' => $contents,
            'headers' => $headers,
            'filename' => $filename,
        ]);

        return $this;
    }

    /**
     * Indicate the request is a multi-part form request.
     *
     * @return $this
     */
    public function asMultipart(): self
    {
        return $this->bodyFormat('multipart');
    }

    /**
     * Specify the body format of the request.
     *
     * @param string $format
     * @return $this
     */
    public function bodyFormat(string $format): self
    {
        return tap($this, function ($request) use ($format) {
            $this->bodyFormat = $format;
        });
    }

    /**
     * Specify the request's content type.
     *
     * @param string $contentType
     * @return $this
     */
    public function contentType(string $contentType): self
    {
        return $this->withHeaders(['Content-Type' => $contentType]);
    }

    /**
     * Indicate that JSON should be returned by the server.
     *
     * @return $this
     */
    public function acceptJson(): self
    {
        return $this->accept('application/json');
    }

    /**
     * Indicate the type of content that should be returned by the server.
     *
     * @param string $contentType
     * @return $this
     */
    public function accept(string $contentType): self
    {
        return $this->withHeaders(['Accept' => $contentType]);
    }

    /**
     * Add the given headers to the request.
     *
     * @param array $headers
     * @return $this
     */
    public function withHeaders(array $headers): self
    {
        return tap($this, function ($request) use ($headers) {
            return $this->options = array_merge_recursive($this->options, [
                'headers' => $headers,
            ]);
        });
    }

    /**
     * Specify the basic authentication username and password for the request.
     *
     * @param string $username
     * @param string $password
     * @return $this
     */
    public function withBasicAuth(string $username, string $password): self
    {
        return tap($this, function ($request) use ($username, $password) {
            return $this->options['auth'] = [$username, $password];
        });
    }

    /**
     * Specify the digest authentication username and password for the request.
     *
     * @param string $username
     * @param string $password
     * @return $this
     */
    public function withDigestAuth(string $username, string $password): self
    {
        return tap($this, function ($request) use ($username, $password) {
            return $this->options['auth'] = [$username, $password, 'digest'];
        });
    }

    /**
     * Specify an authorization token for the request.
     *
     * @param string $token
     * @param string $type
     * @return $this
     */
    public function withToken(string $token, string $type = 'Bearer'): self
    {
        return tap($this, function ($request) use ($token, $type) {
            return $this->options['headers']['Authorization'] = trim($type.' '.$token);
        });
    }

    /**
     * Specify the user agent for the request.
     *
     * @param string $userAgent
     * @return $this
     */
    public function withUserAgent(string $userAgent): self
    {
        return tap($this, function ($request) use ($userAgent) {
            return $this->options['headers']['User-Agent'] = trim($userAgent);
        });
    }

    /**
     * Specify the cookies that should be included with the request.
     *
     * @param array $cookies
     * @param string $domain
     * @return $this
     */
    public function withCookies(array $cookies, string $domain): self
    {
        return tap($this, function ($request) use ($cookies, $domain) {
            return $this->options = array_merge_recursive($this->options, [
                'cookies' => CookieJar::fromArray($cookies, $domain),
            ]);
        });
    }

    /**
     * Indicate that redirects should not be followed.
     *
     * @return $this
     */
    public function withoutRedirecting(): self
    {
        return tap($this, function ($request) {
            return $this->options['allow_redirects'] = false;
        });
    }

    /**
     * Indicate that TLS certificates should not be verified.
     *
     * @return $this
     */
    public function withoutVerifying(): self
    {
        return tap($this, function ($request) {
            return $this->options['verify'] = false;
        });
    }

    /**
     * Specify the path where the body of the response should be stored.
     *
     * @param string|resource $to
     * @return $this
     */
    public function sink($to): self
    {
        return tap($this, function ($request) use ($to) {
            return $this->options['sink'] = $to;
        });
    }

    /**
     * Specify the timeout (in seconds) for the request.
     *
     * @param int $seconds
     * @return $this
     */
    public function timeout(int $seconds): self
    {
        return tap($this, function () use ($seconds) {
            $this->options['timeout'] = $seconds;
        });
    }

    /**
     * Specify the number of times the request should be attempted.
     *
     * @param int $times
     * @param int $sleep
     * @param callable|null $when
     * @return $this
     */
    public function retry(int $times, int $sleep = 0, ?callable $when = null): self
    {
        $this->tries = $times;
        $this->retryDelay = $sleep;
        $this->retryWhenCallback = $when;

        return $this;
    }

    /**
     * Merge new options into the client.
     *
     * @param array $options
     * @return $this
     */
    public function withOptions(array $options): self
    {
        return tap($this, function ($request) use ($options) {
            return $this->options = array_merge_recursive($this->options, $options);
        });
    }

    /**
     * Add new middleware the client handler stack.
     *
     * @param callable $middleware
     * @return $this
     */
    public function withMiddleware(callable $middleware): self
    {
        $this->middleware->push($middleware);

        return $this;
    }

    /**
     * Add a new "before sending" callback to the request.
     *
     * @param callable $callback
     * @return $this
     */
    public function beforeSending($callback): self
    {
        return tap($this, function () use ($callback) {
            $this->beforeSendingCallbacks[] = $callback;
        });
    }

    /**
     * Dump the request before sending.
     *
     * @return $this
     */
    public function dump(): self
    {
        $values = func_get_args();

        return $this->beforeSending(function (Request $request, array $options) use ($values) {
            foreach (array_merge($values, [$request, $options]) as $value) {
                VarDumper::dump($value);
            }
        });
    }

    /**
     * Dump the request before sending and end the script.
     *
     * @return $this
     */
    public function dd(): self
    {
        $values = func_get_args();

        return $this->beforeSending(function (Request $request, array $options) use ($values) {
            foreach (array_merge($values, [$request, $options]) as $value) {
                VarDumper::dump($value);
            }

            exit(1);
        });
    }

    /**
     * Issue a GET request to the given URL.
     *
     * @param string $url
     * @param array|string|null $query
     * @return Response
     * @throws Exception
     */
    public function get(string $url, $query = null)
    {
        return $this->send('GET', $url, [
            'query' => $query,
        ]);
    }

    /**
     * Issue a HEAD request to the given URL.
     *
     * @param string $url
     * @param array|string|null $query
     * @return Response|PromiseInterface
     * @throws Exception
     */
    public function head(string $url, $query = null)
    {
        return $this->send('HEAD', $url, [
            'query' => $query,
        ]);
    }

    /**
     * Issue a POST request to the given URL.
     *
     * @param string $url
     * @param array $data
     * @return Response|PromiseInterface
     * @throws Exception
     */
    public function post(string $url, array $data = [])
    {
        return $this->send('POST', $url, [
            $this->bodyFormat => $data,
        ]);
    }

    /**
     * Issue a PATCH request to the given URL.
     *
     * @param string $url
     * @param array $data
     * @return Response|PromiseInterface
     * @throws Exception
     */
    public function patch(string $url, array $data = [])
    {
        return $this->send('PATCH', $url, [
            $this->bodyFormat => $data,
        ]);
    }

    /**
     * Issue a PUT request to the given URL.
     *
     * @param string $url
     * @param array $data
     * @return Response|PromiseInterface
     * @throws Exception
     */
    public function put(string $url, array $data = [])
    {
        return $this->send('PUT', $url, [
            $this->bodyFormat => $data,
        ]);
    }

    /**
     * Issue a DELETE request to the given URL.
     *
     * @param string $url
     * @param array $data
     * @return Response|PromiseInterface
     * @throws Exception
     */
    public function delete(string $url, array $data = [])
    {
        return $this->send(
            'DELETE',
            $url,
            empty($data) ? [] : [
                $this->bodyFormat => $data,
            ]
        );
    }

    /**
     * Send a pool of asynchronous requests concurrently.
     *
     * @param callable $callback
     * @return array
     */
    public function pool(callable $callback): array
    {
        $results = [];

        $requests = tap(new Pool($this->factory), $callback)->getRequests();

        foreach ($requests as $key => $item) {
            $results[$key] = $item instanceof static && $item->getPromise() ? $item->getPromise()->wait() : $item->wait();
        }

        return $results;
    }

    /**
     * Send the request to the given URL.
     *
     * @param string $method
     * @param string $url
     * @param array $options
     *
     * @return PromiseInterface|Response
     * @throws ConnectionException
     */
    public function send(string $method, string $url, array $options = [])
    {
        $url = ltrim(rtrim($this->baseUrl, '/').'/'.ltrim($url, '/'), '/');

        if (isset($options[$this->bodyFormat])) {
            if ($this->bodyFormat === 'multipart') {
                $options[$this->bodyFormat] = $this->parseMultipartBodyFormat($options[$this->bodyFormat]);
            } elseif ($this->bodyFormat === 'body') {
                $options[$this->bodyFormat] = $this->pendingBody;
            }

            if (is_array($options[$this->bodyFormat])) {
                $options[$this->bodyFormat] = array_merge(
                    $options[$this->bodyFormat],
                    $this->pendingFiles
                );
            }
        } else {
            $options[$this->bodyFormat] = $this->pendingBody;
        }

        [$this->pendingBody, $this->pendingFiles] = [null, []];

        if ($this->async) {
            return $this->makePromise($method, $url, $options);
        }

        return retry($this->tries, function () use ($method, $url, $options) {
            try {
                return tap(new Response($this->sendRequest($method, $url, $options)), function ($response) {
                    $this->populateResponse($response);

                    if ($this->tries > 1 && ! $response->successful()) {
                        $response->throw();
                    }

                    $this->dispatchResponseReceivedEvent($response);
                });
            } catch (ConnectException $e) {
                $this->dispatchConnectionFailedEvent();

                throw new ConnectionException($e->getMessage(), 0, $e);
            }
        }, $this->retryDelay);
    }

    /**
     * Parse multi-part form data.
     *
     * @param array $data
     * @return array|array[]
     */
    protected function parseMultipartBodyFormat(array $data): array
    {
        return collect($data)->map(function ($value, $key) {
            return is_array($value) ? $value : ['name' => $key, 'contents' => $value];
        })->values()->all();
    }

    /**
     * Send an asynchronous request to the given URL.
     *
     * @param string $method
     * @param string $url
     * @param array $options
     * @return PromiseInterface
     * @throws Exception
     */
    protected function makePromise(string $method, string $url, array $options = []): PromiseInterface
    {
        /** @var PromiseInterface $promise */
        $promise = $this->sendRequest($method, $url, $options);

        return $this->promise = $promise
            ->then(function (MessageInterface $message) {
                return tap(new Response($message), function ($response) {
                    $this->populateResponse($response);
                    $this->dispatchResponseReceivedEvent($response);
                });
            })
            ->otherwise(function (TransferException $e) {
                return $e instanceof RequestException && $e->getResponse() ? $this->populateResponse(new Response($e->getResponse())) : $e;
            });
    }

    /**
     * Send a request either synchronously or asynchronously.
     *
     * @param string $method
     * @param string $url
     * @param array $options
     * @return MessageInterface|PromiseInterface
     *
     * @throws Exception
     */
    protected function sendRequest(string $method, string $url, array $options = [])
    {
        $clientMethod = $this->async ? 'requestAsync' : 'request';

        $laravelData = $this->parseRequestData($method, $url, $options);

        return $this->buildClient()->$clientMethod(
            $method,
            $url,
            $this->mergeOptions([
                'laravel_data' => $laravelData,
                'on_stats' => function ($transferStats) {
                    $this->transferStats = $transferStats;
                },
            ], $options)
        );
    }

    /**
     * Get the request data as an array so that we can attach it to the request for convenient assertions.
     *
     * @param string $method
     * @param string $url
     * @param array $options
     * @return array
     */
    protected function parseRequestData(string $method, string $url, array $options)
    {
        $laravelData = $options[$this->bodyFormat] ?? $options['query'] ?? [];

        $urlString = Str::of($url);

        if (empty($laravelData) && $method === 'GET' && $urlString->contains('?')) {
            $laravelData = (string) $urlString->after('?');
        }

        if (is_string($laravelData)) {
            parse_str($laravelData, $parsedData);

            $laravelData = is_array($parsedData) ? $parsedData : [];
        }

        return $laravelData;
    }

    /**
     * Populate the given response with additional data.
     *
     * @param Response $response
     * @return Response
     */
    protected function populateResponse(Response $response): Response
    {
        $response->cookies = $this->cookies; // todo check à quoi ça sert, comment ça marche

        $response->transferStats = $this->transferStats; // todo check à quoi ça sert, comment ça marche

        return $response;
    }

    /**
     * Build the Guzzle client.
     *
     * @return Client
     */
    public function buildClient(): Client
    {
        return $this->requestsReusableClient()
            ? $this->getReusableClient()
            : $this->createClient($this->buildHandlerStack());
    }

    /**
     * Determine if a reusable client is required.
     *
     * @return bool
     */
    protected function requestsReusableClient(): bool
    {
        return ! is_null($this->client) || $this->async;
    }

    /**
     * Retrieve a reusable Guzzle client.
     *
     * @return Client
     */
    protected function getReusableClient(): Client
    {
        return $this->client = $this->client ?: $this->createClient($this->buildHandlerStack());
    }

    /**
     * Create new Guzzle client.
     *
     * @param HandlerStack $handlerStack
     * @return Client
     */
    public function createClient(HandlerStack $handlerStack): Client
    {
        return new Client([
            'handler' => $handlerStack,
            'cookies' => true,
        ]);
    }

    /**
     * Build the Guzzle client handler stack.
     *
     * @return HandlerStack
     */
    public function buildHandlerStack(): HandlerStack
    {
        return $this->pushHandlers(HandlerStack::create());
    }

    /**
     * Add the necessary handlers to the given handler stack.
     *
     * @param HandlerStack $handlerStack
     * @return HandlerStack
     */
    public function pushHandlers(HandlerStack $handlerStack): HandlerStack
    {
        return tap($handlerStack, function ($stack) {
            $stack->push($this->buildBeforeSendingHandler());
            $stack->push($this->buildRecorderHandler());
            $stack->push($this->buildStubHandler());

            $this->middleware->each(function ($middleware) use ($stack) {
                $stack->push($middleware);
            });
        });
    }

    /**
     * Build the before sending handler.
     *
     * @return Closure
     */
    public function buildBeforeSendingHandler(): Closure
    {
        return function ($handler) {
            return function ($request, $options) use ($handler) {
                return $handler($this->runBeforeSendingCallbacks($request, $options), $options);
            };
        };
    }

    /**
     * Build the recorder handler.
     *
     * @return Closure
     */
    public function buildRecorderHandler(): Closure
    {
        return function ($handler) {
            return function ($request, $options) use ($handler) {
                $promise = $handler($request, $options);

                return $promise->then(function ($response) use ($request, $options) {
                    optional($this->factory)->recordRequestResponsePair(
                        (new Request($request))->withData($options['laravel_data']),
                        new Response($response)
                    );

                    return $response;
                });
            };
        };
    }

    /**
     * Build the stub handler.
     *
     * @return Closure
     */
    public function buildStubHandler(): Closure
    {
        return function ($handler) {
            return function ($request, $options) use ($handler) {
                $response = ($this->stubCallbacks ?? collect())
                    ->map(function ($stubCallback) use ($request, $options) {
                        return $stubCallback->__invoke(
                            (new Request($request))->withData($options['laravel_data']),
                            $options
                        );
                    })->filter()
                    ->first();

                if (is_null($response)) {
                    return $handler($request, $options);
                }

                $response = is_array($response) ? Factory::response($response) : $response;

                $sink = $options['sink'] ?? null;

                if ($sink) {
                    $response->then($this->sinkStubHandler($sink));
                }

                return $response;
            };
        };
    }

    /**
     * Get the sink stub handler callback.
     *
     * @param mixed $sink
     * @return Closure
     */
    protected function sinkStubHandler($sink): Closure
    {
        return function ($response) use ($sink) {
            $body = $response->getBody()->getContents();

            if (is_string($sink)) {
                file_put_contents($sink, $body);

                return;
            }

            fwrite($sink, $body);
            rewind($sink);
        };
    }

    /**
     * Execute the "before sending" callbacks.
     *
     * @param RequestInterface $request
     * @param array $options
     * @return mixed
     */
    public function runBeforeSendingCallbacks(RequestInterface $request, array $options)
    {
        return tap($request, function ($request) use ($options) {
            $this->beforeSendingCallbacks->each(function ($callback) use ($request, $options) {
                return $callback->__invoke(
                    (new Request($request))->withData($options['laravel_data']),
                    $options,
                    $this
                );
            });
        });
    }

    /**
     * Merge the given options with the current request options.
     *
     * @param array $options
     * @return array
     */
    public function mergeOptions(...$options): array
    {
        return array_merge_recursive($this->options, ...$options);
    }

    /**
     * Register a stub callable that will intercept requests and be able to return stub responses.
     *
     * @param mixed $callback
     * @return $this
     */
    public function stub($callback): self
    {
        $this->stubCallbacks = collect($callback);

        return $this;
    }

    /**
     * Toggle asynchronicity in requests.
     *
     * @param bool $async
     * @return $this
     */
    public function async(bool $async = true): self
    {
        $this->async = $async;

        return $this;
    }

    /**
     * Retrieve the pending request promise.
     *
     * @return PromiseInterface|null
     */
    public function getPromise(): ?PromiseInterface
    {
        return $this->promise;
    }

    /**
     * Dispatch the RequestSending event if a dispatcher is available.
     *
     * @return void
     */
    protected function dispatchRequestSendingEvent(): void
    {
        if (($dispatcher = optional($this->factory)->getDispatcher()) && $this->request) {
            $dispatcher->dispatch(new RequestSending($this->request));
        }
    }

    /**
     * Dispatch the ResponseReceived event if a dispatcher is available.
     *
     * @param Response $response
     * @return void
     */
    protected function dispatchResponseReceivedEvent(Response $response)
    {
        if (
            ! ($dispatcher = optional($this->factory)->getDispatcher()) ||
            ! $this->request
        ) {
            return;
        }

        $dispatcher->dispatch(new ResponseReceived($this->request));
    }

    /**
     * Dispatch the ConnectionFailed event if a dispatcher is available.
     *
     * @return void
     */
    protected function dispatchConnectionFailedEvent(): void
    {
        if (($dispatcher = optional($this->factory)->getDispatcher()) && $this->request) {
            $dispatcher->dispatch(new ConnectionFailed($this->request));
        }
    }

    /**
     * Set the client instance.
     *
     * @param Client $client
     * @return $this
     */
    public function setClient(Client $client): self
    {
        $this->client = $client;

        return $this;
    }

    /**
     * Create a new client instance using the given handler.
     *
     * @param callable $handler
     * @return $this
     */
    public function setHandler($handler): self
    {
        $this->client = $this->createClient(
            $this->pushHandlers(HandlerStack::create($handler))
        );

        return $this;
    }
}
