<?php

namespace Inside\Services;

use Composer\Factory;
use Composer\IO\BufferIO;
use Composer\Json\JsonFile;
use Composer\Package\Locker;
use Composer\Package\RootPackageInterface;
use Exception;
use Generator;
use Illuminate\Pipeline\Pipeline;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Inside\Application;
use Inside\Facades\Inside;
use Inside\Support\BackofficeEntry;
use Inside\Support\InsidePackage;
use Inside\Support\Requirements\FrontRequirement;
use Inside\Support\Requirements\OptionRequirement;
use Inside\Support\Requirements\RequirementNameResolver;
use Inside\Support\Requirements\SortedRequirements;
use Psr\SimpleCache\InvalidArgumentException;
use Symfony\Component\Finder\Finder;

/**
 * Inside Package Service
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class PackageService
{
    /**
     * In memory package list
     *
     * @var Collection<InsidePackage>
     */
    protected Collection $packages;

    protected string $cacheKey;

    protected array $requirements = [
        'option' => OptionRequirement::class,
        'front' => FrontRequirement::class,
    ];

    protected array $requirementGroups = [];

    protected array $requirementsPriority = [
        FrontRequirement::class,
        OptionRequirement::class,
    ];

    /**
     * @throws Exception
     * @throws InvalidArgumentException
     */
    public function __construct(
        protected Application $app
    ) {
        $this->cacheKey = 'inside.'.md5(config('app.key')).'.package.list';
        $this->packages = $this->loadPackageList();
    }

    /**
     * List all inside installed packages
     *
     * @return Collection<InsidePackage>
     */
    public function list(): Collection
    {
        return $this->packages;
    }

    /**
     * Get package information
     *
     * @param  string  $name
     * @return InsidePackage|null
     */
    public function get(string $name): ?InsidePackage
    {
        $name = $this->getPackageName($name);
        if (! $this->has($name)) {
            return null;
        }

        return $this->packages[$name];
    }

    /**
     * has package ?
     *
     * @param  string  $name
     * @return bool
     */
    public function has(string $name): bool
    {
        return $this->packages->has($this->getPackageName($name));
    }

    /**
     * get Package name
     *
     * @param  string  $name
     * @return string
     */
    protected function getPackageName(string $name): string
    {
        if (! Str::startsWith($name, 'maecia/')) {
            return 'maecia/'.$name;
        }

        return $name;
    }

    /**
     * @return array
     */
    public function getRequirements(): array
    {
        return $this->requirements;
    }

    /**
     * @return array
     */
    public function getRequirementGroups(): array
    {
        return $this->requirementGroups;
    }

    /**
     * @return array
     */
    public function getRequirementsPriority(): array
    {
        return $this->requirementsPriority;
    }

    /**
     * Load package list from cache if set or reload cache if not
     *
     * @return Collection<InsidePackage>
     * @throws Exception|InvalidArgumentException
     */
    protected function loadPackageList(): Collection
    {
        if (! Inside::isInstalled()) {
            return collect();
        }
        if (! Cache::store('file')->has($this->cacheKey)
            || ! Cache::store('file')->get($this->cacheKey) instanceof Collection
        ) {
            if (! getenv('COMPOSER_HOME')) {
                putenv('COMPOSER_HOME='.cms_base_path('storage/app/tmp'));
            }
            if (File::exists(cms_base_path('vendor/composer/installed.json'))) {
                $content = json_decode(file_get_contents(cms_base_path('vendor/composer/installed.json')), true);
                $versions = collect();
                foreach ($content['packages'] as $package) {
                    $versions[$package['name']] = $package['version'].'@'.($package['source']['reference'] ??
                            $package['dist']['reference'] ?? '');
                }
            } elseif (File::exists(cms_base_path('composer.lock'))) {
                $composer = Factory::create(new BufferIO(), cms_base_path('composer.json'), false);
                $rootPackage = $composer->getPackage();
                $locker = $composer->getLocker();
                if (! $locker instanceof Locker) {
                    throw new Exception('Can not get composer locker!');
                }
                /** @phpstan-ignore-next-line */
                $versions = iterator_to_array(self::getVersions($composer->getLocker(), $rootPackage));
            } else {
                throw new Exception('Nor vendor/composer/installed.json nor composer.lock exists');
            }

            $packages = collect();
            foreach ($versions as $package => $version) {
                // Load inside modules
                if (! Str::startsWith($package, 'maecia/')) {
                    continue;
                }
                if (Str::endsWith($package, '-front')) {
                    continue;
                }
                $type = 'inside';
                $jsonPath = cms_base_path('vendor/'.$package.'/composer.json');
                if (! File::exists($jsonPath)) {
                    // Try a drupal module
                    $jsonPath =
                        cms_base_path('modules/custom/'.str_replace('maecia/', '', $package).'/composer.json');
                    if (File::exists($jsonPath)) {
                        $type = 'drupal-module';
                    } else {
                        continue;
                    }
                }
                $jsonFile = new JsonFile($jsonPath);
                try {
                    $composer = $jsonFile->read();
                    $insidePackage = new InsidePackage($package, $version, $composer['autoload']['psr-4'] ?? [], $type);

                    if (isset($composer['extra']) && isset($composer['extra']['laravel'])) {
                        $insidePackage->setProviders($this->extractProviders($composer['extra']['laravel']));
                    }
                    if ($type == 'inside') {
                        if (isset($composer['extra']) && isset($composer['extra']['inside'])) {
                            $insidePackage->setLoadedBackofficeEntries(
                                $this->extractSourceBackofficeEntries($composer['extra']['inside'])
                            );
                            $insidePackage->setBackofficeEntries(
                                $this->extractBackofficeEntries($insidePackage, $composer['extra']['inside'])
                            );
                            $insidePackage->setDependencies(
                                $this->extractDependencies($composer['extra']['inside'])
                            );
                        }
                        $insidePackage->setRoutes($this->loadRoutes($package));
                        $insidePackage->setChannels($this->loadChannels($package));
                        $insidePackage->setConsoles($this->loadConsole($package));
                    }
                    $packages[$package] = $insidePackage;
                } catch (\Seld\JsonLint\ParsingException $e) {
                    Log::error($e->getMessage(), ['exception' => $e]);
                }
            }

            $this->packageComputePackageOrderByDependencies($packages);
            $packages = $packages->sortBy(
                function (InsidePackage $package) {
                    return $package->getOrder();
                }
            );
            Cache::store('file')->forever($this->cacheKey, $packages);

            return $packages;
        }

        return Cache::store('file')->get($this->cacheKey);
    }

    /**
     * Extract all composer's extra providers
     *
     * @param array $extra
     * @return array
     */
    private function extractProviders(array $extra): array
    {
        $providers = [];
        if (isset($extra['providers'])) {
            foreach ($extra['providers'] as $provider) {
                $providers[] = $provider;
            }
        }

        return $providers;
    }

    /**
     * @param  array  $extra
     * @return array
     */
    protected function extractDependencies(array $extra): array
    {
        $dependencies = [];

        if (isset($extra['dependencies'])) {
            foreach ($extra['dependencies'] as $dependency) {
                if (! Str::startsWith($dependency, 'maecia/')) {
                    $dependency = 'maecia/'.$dependency;
                }
                $dependencies[] = $dependency;
            }
        }

        return $dependencies;
    }

    private function extractSourceBackofficeEntries(array $insideExtra): array
    {
        $entries = [];

        if (isset($insideExtra['backoffice_entries'])) {
            foreach ($insideExtra['backoffice_entries'] as $entry) {
                $entries[] = new BackofficeEntry($entry);
            }
        }

        return $entries;
    }

    /**
     * @param  InsidePackage  $insidePackage
     * @param  array  $insideExtra
     * @return array
     * @throws Exception
     */
    private function extractBackofficeEntries(InsidePackage $insidePackage, array $insideExtra): array
    {
        $entries = [];

        if (isset($insideExtra['backoffice_entries'])) {
            foreach ($insideExtra['backoffice_entries'] as $entry) {
                $entry = new BackofficeEntry($entry);
                $requirements = $this->gatherRequirementsForBackofficeEntry($entry);
                $meetsRequirements = (new Pipeline($this->app))
                    ->send($insidePackage)
                    ->through($requirements)
                    ->then(function ($entry) {
                        return true;
                    });
                if ($meetsRequirements) {
                    $entries[] = $entry->getName();
                }
            }
        }

        return $entries;
    }

    private function loadExternal(string $package, string $name): array
    {
        $finder = new Finder();
        $externals = [];
        $path = cms_base_path('vendor/'.$package);
        $depth = '<= 3';

        // inside has subfolders with external + vendor, so we should load these files without the ones in the inside vendors
        if ($package == 'maecia/inside') {
            $path = $path.'/*';
            $depth = '<= 1';
        }

        foreach (
            $finder->in($path)->files()->name($name.'.php')->depth($depth)->followLinks() as $external
        ) {
            $externals[] = $external->getRealPath();
        }

        return $externals;
    }

    /**
     * @param  BackofficeEntry  $entry
     * @return array
     * @throws Exception
     */
    public function gatherRequirementsForBackofficeEntry(BackofficeEntry $entry): array
    {
        $requirements = collect($entry->getRequirements())->map(function ($requirement) {
            return (array) RequirementNameResolver::resolve(
                $requirement,
                $this->getRequirements(),
                $this->getRequirementGroups()
            );
        })->flatten()->reject(function ($name) {
            if ($name instanceof \Closure) {
                return false;
            }

            if (! class_exists($name)) {
                return false;
            }
        })->values();

        return $this->sortRequirements($requirements);
    }

    /**
     * @param Collection $requirements
     * @return array
     */
    protected function sortRequirements(Collection $requirements): array
    {
        return (new SortedRequirements($this->getRequirementsPriority(), $requirements))->all();
    }

    private function loadRoutes(string $package): array
    {
        return $this->loadExternal($package, 'routes');
    }

    private function loadChannels(string $package): array
    {
        return $this->loadExternal($package, 'channels');
    }

    private function loadConsole(string $package): array
    {
        return $this->loadExternal($package, 'console');
    }

    /**
     * Determine installed version of all packages using composer.lock
     *
     * @param  Locker  $locker
     * @param  RootPackageInterface  $rootPackage
     * @return Generator
     */
    private static function getVersions(Locker $locker, RootPackageInterface $rootPackage): Generator
    {
        $lockData = $locker->getLockData();

        $lockData['packages-dev'] = $lockData['packages-dev'] ?? [];

        foreach (array_merge($lockData['packages'], $lockData['packages-dev']) as $package) {
            yield $package['name'] => $package['version'].'@'.($package['source']['reference'] ??
                    $package['dist']['reference'] ?? '');
        }

        foreach ($rootPackage->getReplaces() as $replace) {
            $version = $replace->getPrettyConstraint();
            if ($version === 'self.version') {
                $version = $rootPackage->getPrettyVersion();
            }

            yield $replace->getTarget() => $version.'@'.$rootPackage->getSourceReference();
        }

        yield $rootPackage->getName() => $rootPackage->getPrettyVersion().'@'.$rootPackage->getSourceReference();
    }

    /**
     * @throws Exception
     */
    public function reload(): void
    {
        Cache::store('file')->forget($this->cacheKey);
        $this->packages = $this->loadPackageList();
    }

    /**
     * @throws Exception
     */
    protected function packageComputePackageOrderByDependencies(Collection $packages): void
    {
        $processed = collect();

        while ($packages->count() > $processed->count()) {
            $solvedSomething = false;
            foreach ($packages as $name => $package) {
                /** @var InsidePackage $package */
                if ($processed->has($name)) {
                    continue;
                }

                if ($package->isPrimary()) {
                    $processed[$name] = $package;
                    $solvedSomething = true;
                    break;
                }
                if ($package->isDeferred()) {
                    continue;
                }
                $resolved = true;
                foreach ($package->getDependencies() as $dependency) {
                    if (! $processed->has($dependency)) {
                        // Dependency missing
                        $resolved = false;
                        break;
                    }
                }
                if ($resolved) {
                    // All dependencies met
                    $processed[$name] = $package;
                    $solvedSomething = true;
                }
            }
            if (! $solvedSomething) {
                // Check deferred
                foreach ($packages as $name => $package) {
                    if ($package->isDeferred()) {
                        $processed[$name] = $package;
                        $solvedSomething = true;
                    }
                }
                if (! $solvedSomething) {
                    throw new Exception(
                        '[PackageService::packageComputePackageOrderByDependencies] we did do solve package dependencies'
                    );
                }
            }
        }
        $order = 1;
        foreach ($processed as $package) {
            $package->setOrder($order++);
        }
    }
}
