<?php

namespace Inside\Services\Monitor\Checkers;

use GuzzleHttp\Client as HttpClient;
use GuzzleHttp\Exception\GuzzleException;
use Illuminate\Support\Str;
use Symfony\Component\HttpFoundation\File\UploadedFile as SymfonyUploadedFile;

trait MakesHttpRequests
{
    protected array $defaultHeaders = [];

    protected array $defaultCookies = [];

    protected array $unencryptedCookies = [];

    protected array $serverVariables = [];

    protected bool $followRedirects = false;

    protected bool $encryptCookies = true;

    protected bool $withCredentials = false;

    public function withHeaders(array $headers): self
    {
        $this->defaultHeaders = array_merge($this->defaultHeaders, $headers);

        return $this;
    }

    public function withHeader(string $name, string $value): self
    {
        $this->defaultHeaders[$name] = $value;

        return $this;
    }

    public function withToken(string $token, string $type = 'Bearer'): self
    {
        return $this->withHeader('Authorization', $type.' '.$token);
    }

    public function flushHeaders(): self
    {
        $this->defaultHeaders = [];

        return $this;
    }

    public function withServerVariables(array $server): self
    {
        $this->serverVariables = $server;

        return $this;
    }

    public function withCookies(array $cookies): self
    {
        $this->defaultCookies = array_merge($this->defaultCookies, $cookies);

        return $this;
    }

    public function withCookie(string $name, string $value): self
    {
        $this->defaultCookies[$name] = $value;

        return $this;
    }

    public function withUnencryptedCookies(array $cookies): self
    {
        $this->unencryptedCookies = array_merge($this->unencryptedCookies, $cookies);

        return $this;
    }

    public function withUnencryptedCookie(string $name, string $value): self
    {
        $this->unencryptedCookies[$name] = $value;

        return $this;
    }

    public function followingRedirects(): self
    {
        $this->followRedirects = true;

        return $this;
    }

    public function withCredentials(): self
    {
        $this->withCredentials = true;

        return $this;
    }

    public function disableCookieEncryption(): self
    {
        $this->encryptCookies = false;

        return $this;
    }

    public function get($uri, array $headers = []): array
    {
        $cookies = $this->prepareCookiesForRequest();

        return $this->call(
            'GET',
            $uri,
            [
                'cookies' => $cookies,
                'headers' => $headers,
            ]
        );
    }

    public function getJson(string $uri, array $headers = []): array
    {
        return $this->json(
            'GET',
            $uri,
            [
                'headers' => $headers,
            ]
        );
    }

    public function post(string $uri, array $data = [], array $headers = []): array
    {
        $cookies = $this->prepareCookiesForRequest();

        return $this->call(
            'POST',
            $uri,
            [
                'form_params' => $data,
                'headers' => $headers,
                'cookies' => $cookies,
            ]
        );
    }

    public function postJson(string $uri, array $data = [], array $headers = []): array
    {
        return $this->json(
            'POST',
            $uri,
            [
                'json' => $data,
                'headers' => $headers,
            ]
        );
    }

    public function put(string $uri, array $data = [], array $headers = []): array
    {
        $cookies = $this->prepareCookiesForRequest();

        return $this->call(
            'PUT',
            $uri,
            [
                'form_params' => $data,
                'headers' => $headers,
                'cookies' => $cookies,
            ]
        );
    }

    public function putJson(string $uri, array $data = [], array $headers = []): array
    {
        return $this->json('PUT', $uri, $data, $headers);
    }

    public function patch(string $uri, array $data = [], array $headers = []): array
    {
        $cookies = $this->prepareCookiesForRequest();

        return $this->call(
            'PATCH',
            $uri,
            [
                'form_params' => $data,
                'cookies' => $cookies,
                'headers' => $headers,
            ]
        );
    }

    public function patchJson(string $uri, array $data = [], array $headers = []): array
    {
        return $this->json('PATCH', $uri, $data, $headers);
    }

    public function delete(string $uri, array $data = [], array $headers = []): array
    {
        $cookies = $this->prepareCookiesForRequest();

        return $this->call(
            'DELETE',
            $uri,
            [
                'query' => $data,
                'cookies' => $cookies,
                'headers' => $headers,
            ]
        );
    }

    public function deleteJson(string $uri, array $data = [], array $headers = []): array
    {
        return $this->json('DELETE', $uri, $data, $headers);
    }

    public function options(string $uri, array $data = [], array $headers = []): array
    {
        $cookies = $this->prepareCookiesForRequest();

        return $this->call(
            'OPTIONS',
            $uri,
            [
                'query' => $data,
                'cookies' => $cookies,
                'headers' => $headers,
            ]
        );
    }

    public function optionsJson(string $uri, array $data = [], array $headers = []): array
    {
        return $this->json('OPTIONS', $uri, $data, $headers);
    }

    public function json(string $method, string $uri, array $data = [], array $headers = []): array
    {
        $content = json_encode($data);

        $headers = array_merge(
            [
                'CONTENT_LENGTH' => mb_strlen($content, '8bit'),
                'CONTENT_TYPE' => 'application/json',
                'Accept' => 'application/json',
            ],
            $headers
        );

        return $this->call(
            $method,
            $uri,
            [
                'cookies' => $this->prepareCookiesForJsonRequest(),
                'form_params' => $data,
                'header' => $headers,
                'json' => $content,
            ]
        );
    }

    /**
     * @throws GuzzleException
     */
    public function call(string $method, string $uri, array $options = []): array
    {
        $client = new  HttpClient();
        $response = $client->request(
            $method,
            $this->prepareUrlForRequest($uri),
            array_merge(
                $options,
                [
                    'http_errors' => false,
                ]
            )
        );

        return $this->prepareResponse($response);
    }

    protected function prepareUrlForRequest(string $uri): string
    {
        if (Str::startsWith($uri, '/')) {
            $uri = substr($uri, 1);
        }

        return trim(url($uri), '/');
    }

    protected function transformHeadersToServerVars(array $headers): array
    {
        return collect(array_merge($this->defaultHeaders, $headers))->mapWithKeys(
            function ($value, $name) {
                $name = strtr(strtoupper($name), '-', '_');

                return [$this->formatServerHeaderKey($name) => $value];
            }
        )->all();
    }

    protected function formatServerHeaderKey(string $name): string
    {
        if (! Str::startsWith($name, 'HTTP_') && $name !== 'CONTENT_TYPE' && $name !== 'REMOTE_ADDR') {
            return 'HTTP_'.$name;
        }

        return $name;
    }

    protected function extractFilesFromDataArray(array &$data): array
    {
        $files = [];

        foreach ($data as $key => $value) {
            if ($value instanceof SymfonyUploadedFile) {
                $files[$key] = $value;

                unset($data[$key]);
            }

            if (is_array($value)) {
                $files[$key] = $this->extractFilesFromDataArray($value);

                $data[$key] = $value;
            }
        }

        return $files;
    }

    protected function prepareCookiesForRequest(): array
    {
        if (! $this->encryptCookies) {
            return array_merge($this->defaultCookies, $this->unencryptedCookies);
        }

        return collect($this->defaultCookies)->map(
            function ($value) {
                return encrypt($value);
            }
        )->merge($this->unencryptedCookies)->all();
    }

    protected function prepareCookiesForJsonRequest(): array
    {
        return $this->withCredentials ? $this->prepareCookiesForRequest() : [];
    }

    protected function followRedirects(mixed $response): array
    {
        while ($response->isRedirect()) {
            $response = $this->get($response->headers->get('Location'));
        }

        $this->followRedirects = false;

        return $this->prepareResponse($response);
    }

    public function decodeResponseJson(mixed $response, ?string $key = null): mixed
    {
        $decodedResponse = json_decode($response->getBody(), true);

        if (is_null($decodedResponse) || $decodedResponse === false) {
            if (isset($response->exception)) {
                throw $response->exception;
            }
        }

        // @phpstan-ignore-next-line
        return data_get($decodedResponse, $key);
    }

    protected function prepareResponse(mixed $response): array
    {
        return [
            'successful' => $response->getStatusCode() == 200,
            'code' => $response->getStatusCode(),
            'response' => $this->decodeResponseJson($response),
        ];
    }
}
