<?php

namespace Inside\Services\Monitor\Checkers;

use Illuminate\Support\Str;
use ReflectionClass;

/**
 * Class Checker
 */
abstract class Checker
{
    public const PENDING = 1;

    public const RUNNING = 2;

    public const SUCCESSFUL = 3;

    public const FAILED = 4;

    protected int $lastCheckStatus = self::PENDING;

    protected array $lastCheckData = [];

    protected array $checkers = [];

    /**
     * is this checker supported
     */
    public function isSupported(): bool
    {
        return true;
    }

    /**
     * Get checker name
     */
    public function getName(): string
    {
        $className = (new ReflectionClass($this))->getShortName();

        return Str::snake(Str::before($className, 'Checker'));
    }

    /**
     *  Preparing check if necessary
     */
    public function prepareCheck(array $checkers): void
    {
        $this->lastCheckStatus = self::RUNNING;
        $this->checkers = $checkers;

        if (method_exists($this, 'getLimit')) {
            $this->lastCheckData['limit'] = $this->getLimit();
        }
    }

    public function getCheckResult(): array
    {
        return [
            'status' => $this->getCheckStatusName($this->getLastCheckStatus()),
            'healthy' => $this->isHealthy(),
            'data' => $this->getLasCheckData(),
        ];
    }

    public function getLastCheckStatus(): int
    {
        return $this->lastCheckStatus;
    }

    public function getLasCheckData(): array
    {
        return $this->lastCheckData;
    }

    protected function failing(string $message): void
    {
        $this->lastCheckData = [
            'error' => $message,
        ];
    }

    protected function getCheckers(): array
    {
        return $this->checkers;
    }

    protected function getCheckStatusName(int $checkStatus): string
    {
        return match ($checkStatus) {
            self::PENDING => 'pending',
            self::RUNNING => 'running',
            self::SUCCESSFUL => 'successful',
            self::FAILED => 'failed',
            default => 'unkown',
        };
    }

    public function runCheck(): void
    {
        if ($this->processRunCheck()) {
            $this->lastCheckStatus = self::SUCCESSFUL;
        } else {
            $this->lastCheckStatus = self::FAILED;
        }
    }
}
