<?php

namespace Inside\Services;

use Closure;
use Exception;
use Illuminate\Contracts\View\View;
use Illuminate\Support\Collection;
use Inside\Facades\Package;

class InsideApplicationService
{
    /**
     * Scripts
     */
    private array $script = [];

    private array $deferredScript = [];

    private array $headerJs = [];

    private array $js = [];

    private array $css = [];

    /**
     * Does inside drupal module is enable
     */
    protected bool $drupalModuleEnable = true;

    /**
     * Get Command intro text
     *
     * @return string
     */
    public function getCommandInfo(): string
    {
        return 'Inside 2 ( Version '.inside_version().' ) ';
    }

    /**
     * Add a css or get all css
     *
     * @param  array|null  $css
     *
     * @return View|array ($css is null ? array : View)
     * @throws Exception
     */
    public function css(?array $css = null): View|array
    {
        if (! is_null($css)) {
            return $this->css = array_merge($this->css, $css);
        }
        $css = array_merge($this->css, inside_mix('css')->toArray());

        $css = array_filter(array_unique($css));

        return view('partials.css', compact('css'));
    }

    /**
     * Add js or get all js
     *
     * @param  string|null  $js
     * @param  array  $parameters
     *
     * @return View|array ($js is null ? array : View)
     * @throws Exception
     */
    public function js(?string $js = null, array $parameters = []): View|array
    {
        if (! is_null($js)) {
            return $this->js = array_merge($this->js, [['src' => $js, 'parameters' => $parameters]]);
        }

        // Load inside theme js
        $insideJs = [];
        foreach (inside_mix('js') as $j) {
            $insideJs[] = ['src' => $j];
        }

        // Add frontend js
        $js = array_merge($insideJs, $this->js);

        $js = array_filter($js, function ($value) {
            return isset($value['src']) && ! empty($value['src']);
        });

        return view('partials.js', compact('js'));
    }

    /**
     * Add header js or get all header js
     *
     * @param  string|null  $js
     * @param  array  $parameters
     *
     * @return View|array ($js is null ? array : View)
     */
    public function headerJs(?string $js = null, array $parameters = []): View|array
    {
        if (! is_null($js)) {
            return $this->headerJs = array_merge($this->headerJs, [['src' => $js, 'parameters' => $parameters]]);
        }

        $js = array_filter($this->headerJs, function ($value) {
            return isset($value['src']) && ! empty($value['src']);
        });

        return view('partials.js', compact('js'));
    }

    /**
     * Add script or get all scripts
     *
     * @param  string|null  $script
     * @param  bool  $deferred
     *
     * @return View|array ($script is null ? array : View)
     */
    public function script(?string $script = '', bool $deferred = false): View|array
    {
        if (! empty($script)) {
            if ($deferred) {
                return $this->deferredScript = array_merge($this->deferredScript, (array) $script);
            }

            return $this->script = array_merge($this->script, (array) $script);
        }

        $script = array_unique(array_merge($this->script, $this->deferredScript));

        return view('partials.script', compact('script'));
    }

    /**
     * Does inside drupal module is enable
     */
    public function isDrupalModuleEnable(): bool
    {
        return $this->drupalModuleEnable;
    }

    /**
     * Disable inside drupal module
     */
    public function disableDrupalModule(): void
    {
        $this->drupalModuleEnable = false;
    }

    /**
     * Enable inside drupal module
     */
    public function enableDrupalModule(): void
    {
        $this->drupalModuleEnable = true;
    }

    /**
     * Work without inside drupal module
     */
    public function withoutDrupalModule(Closure $callback): mixed
    {
        $this->disableDrupalModule();

        try {
            return $callback();
        } finally {
            $this->enableDrupalModule();
        }
    }

    /**
     * Inside is installed
     */
    public function isInstalled(): bool
    {
        if (file_exists(storage_path('framework/inside_installed'))) {
            return true;
        }

        return false;
    }

    /**
     * get All Backoffice Entries
     */
    public function getAllBackofficeEntries(): Collection
    {
        $entries = collect();
        foreach (Package::list() as $package) {
            $entries = $entries->merge($package->getBackofficeEntries());
        }

        return $entries;
    }
}
