<?php

namespace Inside\Services;

use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Intervention\Image\ImageManagerStatic as Image;
use PHP_ICO;

class FaviconsGeneratorService
{
    const PATH = 'assets/favicons';

    const DEFAULT_FAVICON = __DIR__.'/../../resources/assets/img/defaultFavicon.png';

    public static function generate(string $original): array
    {
        $destination = storage_path('app/'.self::PATH);

        if (! File::exists($destination)) {
            File::makeDirectory($destination, 0755, true);
        }

        File::copy($original, $destination.'/original.png');
        $img = Image::make($original);

        $icons = [
            'android-chrome-192x192.png' => [192, 192],
            'android-chrome-512x512.png' => [512, 512],
            'apple-touch-icon.png' => [180, 180],
            'favicon-16x16.png' => [16, 16],
            'favicon-32x32.png' => [32, 32],
            'favicon-48x48.png' => [48, 48],
        ];

        foreach ($icons as $filename => $dimensions) {
            $imgResized = clone $img;
            $imgResized->resize($dimensions[0], $dimensions[1], function ($constraint) {
                $constraint->aspectRatio();
                $constraint->upsize();
            });

            $imgResized->save($destination.'/'.$filename);
        }

        $ico = new PHP_ICO();
        $ico->add_image($destination.'/favicon-16x16.png', [16, 16]);
        $ico->add_image($destination.'/favicon-32x32.png', [32, 32]);
        $ico->add_image($destination.'/favicon-48x48.png', [48, 48]);
        $ico->save_ico($destination.'/favicon.ico');

        $manifest = [
            'name' => config('app.name'),
            'short_name' => config('app.name'),
            'icons' => [
                [
                    'src' => '/favicons/android-chrome-192x192.png',
                    'sizes' => '192x192',
                    'type' => 'image/png',
                ],
                [
                    'src' => '/favicons/android-chrome-512x512.png',
                    'sizes' => '512x512',
                    'type' => 'image/png',
                ],
            ],
            'theme_color' => '#ffffff',
            'background_color' => '#ffffff',
            'display' => 'standalone',
        ];

        File::put($destination.'/site.webmanifest', json_encode($manifest, JSON_PRETTY_PRINT));

        return [
            'original.png' => url(self::PATH.'/original.png'),
            'android-chrome-192x192.png' => url(self::PATH.'/android-chrome-192x192.png'),
            'android-chrome-512x512.png' => url(self::PATH.'/android-chrome-512x512.png'),
            'apple-touch-icon.png' => url(self::PATH.'/apple-touch-icon.png'),
            'favicon-16x16.png' => url(self::PATH.'/favicon-16x16.png'),
            'favicon-32x32.png' => url(self::PATH.'/favicon-32x32.png'),
            'favicon-48x48.png' => url(self::PATH.'/favicon-48x48.png'),
            'favicon.ico' => url(self::PATH.'/favicon.ico'),
            'site.webmanifest' => url(self::PATH.'/site.webmanifest'),
        ];
    }
}
