<?php

namespace Inside\Providers;

use Closure;
use Illuminate\Console\Application as Artisan;
use Illuminate\Console\Command;
use Illuminate\Support\ServiceProvider as BaseServiceProvider;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;
use ReflectionClass;

/**
 * Inside Service provider.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 *
 * @property \Inside\Application $app
 */
class ServiceProvider extends BaseServiceProvider
{
    protected array $bootingCallbacks = [];

    protected array $bootedCallbacks = [];

    protected array $commands = [];

    /**
     * Recursively merge the given configuration with the existing configuration.
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    protected function mergeRecursiveConfigFrom(string $path, string $key): void
    {
        $config = $this->app['config']->get($key, []);

        $this->app['config']->set($key, array_merge_recursive(require $path, $config));
    }

    /**
     * Merge $path to config
     *
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    protected function mergeConfigTo(string $path, string $key): void
    {
        $config = $this->app['config']->get($key, []);

        $this->app['config']->set($key, array_merge($config, require $path));
    }

    /**
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    protected function overrideConfig(string $path, string $key): void
    {
        $config = $this->app['config']->get($key, []);

        $overrides = require $path;
        foreach ($config as $configKey => $configValue) {
            if (isset($overrides[$configKey])) {
                $config[$configKey] = $this->overrideConfigToDefault($overrides[$configKey], $configValue);
            }
        }

        $this->app['config']->set($key, $config);
    }

    /**
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    protected function overrideConfigToDefault(mixed $override, mixed $default): mixed
    {
        $config = $default;
        if (is_array($default)) {
            foreach ($default as $configKey => $configValue) {
                if (is_array($override) && isset($override[$configKey])) {
                    $config[$configKey] = $this->overrideConfigToDefault($override[$configKey], $configValue);
                }
            }
        } else {
            $config = $override;
        }

        return $config;
    }

    public function booting(Closure $callback): void
    {
        $this->bootingCallbacks[] = $callback;
    }

    public function booted(Closure $callback): void
    {
        $this->bootedCallbacks[] = $callback;
    }

    public function callBootingCallbacks(): void
    {
        foreach ($this->bootingCallbacks as $callback) {
            $this->app->call($callback);
        }
    }

    public function callBootedCallbacks(): void
    {
        foreach ($this->bootedCallbacks as $callback) {
            $this->app->call($callback);
        }
        if ($this->app->runningInConsole()) {
            $this->registerCommands();
        }
    }

    public function registerCommands(): void
    {
        foreach ($this->commands as $command) {
            if (is_subclass_of($command, Command::class) && ! (new ReflectionClass($command))->isAbstract()) {
                Artisan::starting(function ($artisan) use ($command) {
                    $artisan->resolve($command);
                });
            }
        }
    }
}
