<?php

namespace Inside\Jobs\Bus;

use Closure;
use DateInterval;
use DateTimeInterface;
use Illuminate\Support\Arr;
use Inside\Jobs\Queue\CallQueuedClosure;
use RuntimeException;

trait Queueable
{
    /**
     * The name of the connection the job should be sent to.
     *
     * @var string|null
     */
    public $connection;

    /**
     * The name of the queue the job should be sent to.
     *
     * @var string|null
     */
    public $queue;

    /**
     * The name of the connection the chain should be sent to.
     *
     * @var string|null
     */
    public $chainConnection;

    /**
     * The name of the queue the chain should be sent to.
     *
     * @var string|null
     */
    public $chainQueue;

    /**
     * The callbacks to be executed on chain failure.
     *
     * @var array|null
     */
    public $chainCatchCallbacks;

    /**
     * The number of seconds before the job should be made available.
     *
     * @var DateTimeInterface|DateInterval|int|null
     */
    public $delay;

    /**
     * Indicates whether the job should be dispatched after all database transactions have committed.
     *
     * @var bool|null
     */
    public $afterCommit;

    /**
     * The middleware the job should be dispatched through.
     *
     * @var array
     */
    public $middleware = [];

    /**
     * The jobs that should run if this job is successful.
     *
     * @var array
     */
    public $chained = [];

    public function onConnection(mixed $connection): self
    {
        $this->connection = $connection;

        return $this;
    }

    public function onQueue(mixed $queue): self
    {
        $this->queue = $queue;

        return $this;
    }

    public function allOnConnection(mixed $connection): self
    {
        $this->chainConnection = $connection;
        $this->connection = $connection;

        return $this;
    }

    public function allOnQueue(mixed $queue): self
    {
        $this->chainQueue = $queue;
        $this->queue = $queue;

        return $this;
    }

    /**
     * @param DateTimeInterface|DateInterval|int|null $delay
     * @return $this
     */
    public function delay($delay): self
    {
        $this->delay = $delay;

        return $this;
    }

    /**
     * @return $this
     */
    public function afterCommit(): self
    {
        $this->afterCommit = true;

        return $this;
    }

    /**
     * @return $this
     */
    public function beforeCommit(): self
    {
        $this->afterCommit = false;

        return $this;
    }

    /**
     * @param mixed $middleware
     * @return $this
     */
    public function through($middleware): self
    {
        $this->middleware = Arr::wrap($middleware);

        return $this;
    }

    /**
     * @param array $chain
     * @return $this
     */
    public function chain($chain): self
    {
        $this->chained = collect($chain)->map(function ($job) {
            return $this->serializeJob($job);
        })->all();

        return $this;
    }

    /**
     * @param mixed $job
     * @return string
     */
    protected function serializeJob($job): string
    {
        if ($job instanceof Closure) {
            if (! class_exists(CallQueuedClosure::class)) {
                throw new RuntimeException(
                    'To enable support for closure jobs, please install the illuminate/queue package.'
                );
            }

            $job = CallQueuedClosure::create($job);
        }

        return serialize($job);
    }

    /**
     * @return void
     */
    public function dispatchNextJobInChain(): void
    {
        if (! empty($this->chained)) {
            dispatch(tap(unserialize(array_shift($this->chained)), function ($next) {
                $next->chained = $this->chained;

                $next->onConnection($next->connection ?: $this->chainConnection);
                $next->onQueue($next->queue ?: $this->chainQueue);

                $next->chainConnection = $this->chainConnection;
                $next->chainQueue = $this->chainQueue;
                $next->chainCatchCallbacks = $this->chainCatchCallbacks;
            }));
        }
    }

    /**
     * @param mixed $e
     * @return void
     */
    public function invokeChainCatchCallbacks($e)
    {
        collect($this->chainCatchCallbacks)->each(function ($callback) use ($e) {
            $callback($e);
        });
    }
}
