<?php

namespace Inside\Jobs\Bus;

use Illuminate\Bus\Dispatcher as BusDispatcher;
use Illuminate\Container\Container;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\Jobs\SyncJob;
use Illuminate\Support\Collection;

class Dispatcher extends BusDispatcher
{
    /**
     * @var Container
     */
    protected $container;

    public function dispatch($command): mixed
    {
        return $this->queueResolver && $this->commandShouldBeQueued($command)
            ? $this->dispatchToQueue($command)
            : $this->dispatchNow($command);
    }

    public function dispatchSync(mixed $command, mixed $handler = null): mixed
    {
        if (
            $this->queueResolver &&
            $this->commandShouldBeQueued($command) &&
            method_exists($command, 'onConnection')
        ) {
            return $this->dispatchToQueue($command->onConnection('sync'));
        }

        return $this->dispatchNow($command, $handler);
    }

    public function dispatchNow($command, $handler = null): mixed
    {
        $uses = class_uses_recursive($command);

        if (
            in_array(InteractsWithQueue::class, $uses) &&
            in_array(Queueable::class, $uses) &&
            ! $command->job
        ) {
            $command->setJob(new SyncJob($this->container, '[]', 'sync', 'sync'));
        }

        if ($handler || $handler = $this->getCommandHandler($command)) {
            $callback = function ($command) use ($handler) {
                $method = method_exists($handler, 'handle') ? 'handle' : '__invoke';

                return $handler->{$method}($command);
            };
        } else {
            $callback = function ($command) {
                $method = method_exists($command, 'handle') ? 'handle' : '__invoke';

                return $this->container->call([$command, $method]);
            };
        }

        return $this->pipeline->send($command)->through($this->pipes)->then($callback);
    }

    public function chain(mixed $jobs): PendingChain
    {
        $jobs = Collection::wrap($jobs);

        return new PendingChain($jobs->shift(), $jobs->toArray());
    }
}
