<?php

namespace Inside\Http\Requests;

use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Contracts\Container\Container;
use Illuminate\Contracts\Validation\ValidatesWhenResolved;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Request;
use Illuminate\Validation\Factory as ValidationFactory;
use Illuminate\Validation\ValidatesWhenResolvedTrait;
use Illuminate\Validation\ValidationException;
use Inside\Facades\Inside;
use Inside\Support\ValidatedInput;
use Inside\Validation\Validator as InsideValidator;

class FormRequest extends Request implements ValidatesWhenResolved
{
    use ValidatesWhenResolvedTrait;

    protected Container $container;

    protected ?InsideValidator $validator = null;

    protected string $errorBag = 'default';

    /**
     * Custom message errors
     */
    public function messages(): array
    {
        return [];
    }

    /**
     * Custom attributesx
     */
    public function attributes(): array
    {
        return [];
    }

    /**
     * Set Validator
     */
    public function setValidator(InsideValidator $validator): self
    {
        $this->validator = $validator;

        return $this;
    }

    /**
     * Set container
     */
    public function setContainer(Container $container): self
    {
        $this->container = $container;

        return $this;
    }

    /**
     * Get Validator instance
     *
     * @throws BindingResolutionException
     */
    protected function getValidatorInstance(): InsideValidator
    {
        if ($this->validator) {
            return $this->validator;
        }

        $factory = $this->container->make(ValidationFactory::class);

        if (method_exists($this, 'validator')) {
            $validator = $this->container->call([$this, 'validator'], compact('factory'));
        } else {
            $validator = $this->createDefaultValidator($factory);
        }

        if (method_exists($this, 'withValidator')) {
            $this->withValidator($validator);
        }

        $this->setValidator($validator);

        return $validator;
    }

    /**
     * Create default validator instance
     */
    protected function createDefaultValidator(ValidationFactory $factory): Validator
    {
        $factory->resolver(
            function ($translator, $data, $rules, $messages, $customAttributes) {
                return new InsideValidator($translator, $data, $rules, $messages, $customAttributes);
            }
        );

        /** @var InsideValidator $validator */
        $validator = $factory->make(
            $this->validationData(),
            $this->container->call([$this, 'rules']),
            $this->messages(),
            $this->attributes()
        );

        return $validator;
    }

    /**
     * Get data to be validated
     */
    public function validationData(): array
    {
        return $this->all();
    }

    /**
     * Handle failed validation
     *
     * @param  Validator  $validator
     * @return void
     *
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator): void
    {
        throw (new ValidationException($validator))->errorBag($this->errorBag);
    }

    /**
     * Get safe validated data
     *
     * @param  array|null  $keys
     * @return array|ValidatedInput
     * @throws BindingResolutionException
     */
    public function safe(array $keys = null): array|ValidatedInput
    {
        if (! is_array($keys)) {
            return $this->getValidatorInstance()->safe();
        }

        /** @var ValidatedInput $validatedInput */
        $validatedInput = $this->getValidatorInstance()->safe();

        return $validatedInput->only($keys);
    }

    /**
     * @throws ValidationException|BindingResolutionException
     */
    public function validated(?string $key = null, mixed $default = null): mixed
    {
        return data_get($this->getValidatorInstance()->validated(), $key ?? [], $default);
    }

    /**
     * @param  Request  $from
     * @param  Request|null  $to
     * @return Request
     */
    public static function createFrom(Request $from, $to = null): Request
    {
        $request = $to ?: new static();

        $files = array_filter($from->files->all());

        $request->initialize(
            $from->query->all(),
            $from->request->all(),
            $from->attributes->all(),
            $from->cookies->all(),
            $files,
            $from->server->all(),
            $from->getContent()
        );

        $request->headers->replace($from->headers->all());

        $request->setJson($from->json());

        if ($from->hasSession()) {
            $request->setLaravelSession($from->session());
        }

        $request->setUserResolver($from->getUserResolver());

        $request->setRouteResolver($from->getRouteResolver());

        return $request;
    }
}
