<?php

namespace Inside\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Inside\Support\ServerTiming;
use Symfony\Component\HttpFoundation\Response;

class ServerPerformanceMonitor
{
    /** @var float|mixed|string */
    protected mixed $start;

    public function __construct(
        protected ServerTiming $timing
    ) {
        $this->start = $this->getRequestStartTime();
    }

    public function handle(Request $request, Closure $next): Response
    {
        if (app()->environment() === 'production') {
            return $next($request);
        }

        $this->timing->setDuration('Bootstrap', $this->getElapsedTimeInMs());

        $this->timing->start('App');

        /** @var Response $response */
        $response = $next($request);

        $this->timing->stop('App');

        $this->timing->stopAllUnfinishedEvents();

        $this->timing->setDuration('Total', $this->getElapsedTimeInMs());

        $response->headers->set('Server-Timing', $this->generateHeaders());

        return $response;
    }

    /**
     * @return float|int
     */
    protected function getElapsedTimeInMs(): float|int
    {
        return (microtime(true) - $this->start) * 1000;
    }

    /**
     * @return float|int
     */
    protected function getRequestStartTime(): float|int
    {
        if (defined('LARAVEL_START')) {
            return LARAVEL_START;
        }

        return $_SERVER['REQUEST_TIME_FLOAT'] ?? microtime(true);
    }

    protected function generateHeaders(): string
    {
        $header = '';

        foreach ($this->timing->events() as $eventName => $duration) {
            $eventNameSlug = Str::slug($eventName);

            $header .= "{$eventNameSlug};desc=\"{$eventName}\";";

            if (! is_null($duration)) {
                $header .= "dur={$duration}";
            }

            $header .= ', ';
        }

        return $header;
    }
}
